// Data Produk
const produkData = {
    'standing-flower': {
        name: 'Standing Flower',
        image: 'http://info.queenflorist.art/assets/standing%20flower%20best.jpg',
        description: 'Karangan bunga berdiri yang elegan untuk berbagai acara formal dan perayaan.'
    },
    'paper-flower': {
        name: 'Paper Flower',
        image: 'http://info.queenflorist.art/assets/paper%20flower.jpg',
        description: 'Bunga kertas artistik yang awet dan cantik untuk dekorasi ruangan.'
    },
    'duka-cita': {
        name: 'Bunga Duka Cita',
        image: 'http://info.queenflorist.art/assets/duka%20cita.jpg',
        description: 'Karangan bunga simpati untuk mengungkapkan belasungkawa.'
    },
    'bouquet': {
        name: 'Bouquet Bunga',
        image: 'http://info.queenflorist.art/assets/bouquet.jpg',
        description: 'Buket bunga segar yang indah untuk orang terkasih.'
    },
    'wedding-car': {
        name: 'Wedding Car Decoration',
        image: 'http://info.queenflorist.art/assets/wedcar.jpeg',
        description: 'Dekorasi bunga untuk mobil pengantin yang mewah dan elegan.'
    },
    'balloon-decoration': {
        name: 'Balloon Decoration',
        image: 'http://info.queenflorist.art/assets/balon%20decoration.jpg',
        description: 'Dekorasi balon yang meriah untuk berbagai pesta dan acara.'
    },
    'kado-rustic': {
        name: 'Kado Rustic',
        image: 'http://info.queenflorist.art/assets/kadorustic.jpeg',
        description: 'Kado dengan gaya rustic yang unik dan berkesan.'
    },
    'balloon-parcel': {
        name: 'Balloon Parcel',
        image: 'http://info.queenflorist.art/assets/ballon%20parcel.jpg',
        description: 'Parcel balon yang lucu dan menarik untuk hadiah spesial.'
    },
    'snack-bouquet': {
        name: 'Snack Bouquet',
        image: 'http://info.queenflorist.art/assets/snack%20bouquet.jpg',
        description: 'Buket snack yang kreatif dan lezat untuk hadiah yang berbeda.'
    },
    'tart-snack': {
        name: 'Tart Snack',
        image: 'http://info.queenflorist.art/assets/tart%20snack.jpg',
        description: 'Tart dari berbagai snack yang unik dan menggugah selera.'
    }
};

// DOM Elements
const navbar = document.getElementById('navbar');
const hamburger = document.querySelector('.hamburger');
const navMenu = document.querySelector('.nav-menu');
const navLinks = document.querySelectorAll('.nav-link');
const kategoriBtns = document.querySelectorAll('.kategori-btn');
const produkGrid = document.getElementById('produk-grid');
const modal = document.getElementById('orderModal');
const modalClose = document.querySelector('.close');
const orderForm = document.getElementById('orderForm');
const scrollTopBtn = document.createElement('div');

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    setupEventListeners();
    createScrollTopButton();
    setupSmoothScroll();
    setupNavbarScroll();
    loadDefaultProducts();
});

// Setup Event Listeners
function setupEventListeners() {
    // Mobile menu toggle
    hamburger.addEventListener('click', function() {
        hamburger.classList.toggle('active');
        navMenu.classList.toggle('active');
    });

    // Close mobile menu when clicking on a link
    navLinks.forEach(link => {
        link.addEventListener('click', function() {
            hamburger.classList.remove('active');
            navMenu.classList.remove('active');
        });
    });

    // Kategori buttons
    kategoriBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const kategori = this.dataset.kategori;
            selectKategori(kategori);
            loadProduk(kategori);
        });
    });

    // Modal close
    modalClose.addEventListener('click', closeModal);
    window.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeModal();
        }
    });

    // Order form submission
    orderForm.addEventListener('submit', handleOrderSubmit);

    // Radio button change for delivery option
    const radioButtons = document.querySelectorAll('input[name="pengiriman"]');
    radioButtons.forEach(radio => {
        radio.addEventListener('change', function() {
            toggleAlamatField(this.value);
        });
    });

    // Escape key to close modal
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && modal.style.display === 'block') {
            closeModal();
        }
    });
}

// Setup Smooth Scroll
function setupSmoothScroll() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                // Dynamic navbar height based on screen size
                const navbarHeight = window.innerWidth <= 768 ? 60 : (window.innerWidth <= 480 ? 55 : 70);
                const offsetTop = target.offsetTop - navbarHeight;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    });
}

// Setup Navbar Scroll Effect
function setupNavbarScroll() {
    let lastScrollTop = 0;
    
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        if (scrollTop > 100) {
            navbar.style.background = 'rgba(255, 255, 255, 0.98)';
            navbar.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.1)';
        } else {
            navbar.style.background = 'rgba(255, 255, 255, 0.95)';
            navbar.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.1)';
        }

        // Show/hide scroll to top button
        if (scrollTop > 500) {
            scrollTopBtn.classList.add('show');
        } else {
            scrollTopBtn.classList.remove('show');
        }
        
        lastScrollTop = scrollTop;
    });
}

// Create Scroll to Top Button
function createScrollTopButton() {
    scrollTopBtn.className = 'scroll-top';
    scrollTopBtn.innerHTML = '<i class="fas fa-arrow-up"></i>';
    scrollTopBtn.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
    document.body.appendChild(scrollTopBtn);
}

// Select Kategori
function selectKategori(kategori) {
    kategoriBtns.forEach(btn => {
        btn.classList.remove('active');
    });
    
    const selectedBtn = document.querySelector(`[data-kategori="${kategori}"]`);
    if (selectedBtn) {
        selectedBtn.classList.add('active');
    }
}

// Load Default Products
function loadDefaultProducts() {
    // Load first category by default
    const firstKategori = kategoriBtns[0].dataset.kategori;
    selectKategori(firstKategori);
    loadProduk(firstKategori);
}

// Load Produk
function loadProduk(kategori) {
    const produk = produkData[kategori];
    if (!produk) return;

    // Clear existing products
    produkGrid.innerHTML = '';

    // Create product card
    const produkCard = createProdukCard(produk, kategori);
    produkGrid.appendChild(produkCard);

    // Smooth scroll to produk grid
    setTimeout(() => {
        produkGrid.scrollIntoView({ 
            behavior: 'smooth', 
            block: 'nearest' 
        });
    }, 100);
}

// Create Produk Card
function createProdukCard(produk, kategori) {
    const card = document.createElement('div');
    card.className = 'produk-card';
    card.style.opacity = '0';
    card.style.transform = 'translateY(20px)';
    
    card.innerHTML = `
        <div class="produk-image-container">
            <img src="${produk.image}" alt="${produk.name}" class="produk-image" 
                 onerror="this.src='https://via.placeholder.com/300x250/e91e63/ffffff?text=Queen+Florist'">
        </div>
        <div class="produk-info">
            <h3 class="produk-name">${produk.name}</h3>
            <p class="produk-description">${produk.description}</p>
            <button class="order-btn" onclick="openOrderModal('${kategori}')">
                <i class="fas fa-shopping-cart"></i>
                Order
            </button>
        </div>
    `;

    // Animate card appearance
    setTimeout(() => {
        card.style.transition = 'all 0.6s ease';
        card.style.opacity = '1';
        card.style.transform = 'translateY(0)';
    }, 100);

    return card;
}

// Open Order Modal
function openOrderModal(kategori) {
    const produk = produkData[kategori];
    if (!produk) return;

    // Set modal content
    document.getElementById('modalProdukImage').src = produk.image;
    document.getElementById('modalProdukName').textContent = produk.name;

    // Reset form
    orderForm.reset();
    document.getElementById('alamatGroup').style.display = 'none';

    // Show modal
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';

    // Focus on first input
    setTimeout(() => {
        document.getElementById('nama').focus();
    }, 300);
}

// Close Modal
function closeModal() {
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

// Toggle Alamat Field
function toggleAlamatField(value) {
    const alamatGroup = document.getElementById('alamatGroup');
    const alamatField = document.getElementById('alamat');
    
    if (value === 'kirim') {
        alamatGroup.style.display = 'block';
        alamatField.required = true;
        
        // Animate appearance
        alamatGroup.style.opacity = '0';
        alamatGroup.style.transform = 'translateY(-10px)';
        setTimeout(() => {
            alamatGroup.style.transition = 'all 0.3s ease';
            alamatGroup.style.opacity = '1';
            alamatGroup.style.transform = 'translateY(0)';
        }, 10);
    } else {
        alamatGroup.style.display = 'none';
        alamatField.required = false;
    }
}

// Handle Order Submit
function handleOrderSubmit(e) {
    e.preventDefault();
    
    const formData = new FormData(orderForm);
    const nama = formData.get('nama');
    const catatan = formData.get('catatan');
    const pengiriman = formData.get('pengiriman');
    const alamat = formData.get('alamat');
    const produkName = document.getElementById('modalProdukName').textContent;
    const produkImage = document.getElementById('modalProdukImage').src;

    // Validate form
    if (!nama.trim()) {
        showError('Nama lengkap harus diisi!');
        return;
    }

    if (pengiriman === 'kirim' && !alamat.trim()) {
        showError('Alamat pengiriman harus diisi!');
        return;
    }

    // Create WhatsApp message
    const message = createWhatsAppMessage({
        nama,
        catatan,
        pengiriman,
        alamat,
        produkName,
        produkImage
    });

    // Send to WhatsApp
    const whatsappUrl = `https://wa.me/6281223968122?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');

    // Close modal and show success message
    closeModal();
    showSuccessMessage();
}

// Create WhatsApp Message
function createWhatsAppMessage(data) {
    let message = `🌸 *PEMESANAN QUEEN FLORIST* 🌸\n\n`;
    message += `📋 *Detail Pesanan:*\n`;
    message += `━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`;
    message += `🏷️ *Produk:* ${data.produkName}\n`;
    message += `👤 *Nama:* ${data.nama}\n`;
    message += `📝 *Catatan:* ${data.catatan || '-'}\n`;
    message += `🚚 *Pengiriman:* ${data.pengiriman === 'ambil' ? 'Ambil Sendiri' : 'Kirim ke Alamat'}\n`;
    
    if (data.pengiriman === 'kirim') {
        message += `📍 *Alamat:* ${data.alamat}\n`;
    }
    
    message += `\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`;
    message += `📎 *Gambar Produk:* ${data.produkImage}\n\n`;
    message += `Mohon konfirmasi ketersediaan dan total harga. Terima kasih! 🙏\n`;
    message += `━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`;
    message += `🌺 Queen Florist - Temukan Bunga Terindah untuk Setiap Momen 🌺`;

    return message;
}

// Show Error Message
function showError(message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error-message';
    errorDiv.innerHTML = `
        <i class="fas fa-exclamation-circle"></i>
        <span>${message}</span>
    `;
    
    // Mobile responsive positioning
    const isMobile = window.innerWidth <= 768;
    const topPosition = isMobile ? 80 : 100;
    const rightPosition = isMobile ? 10 : 20;
    const maxWidth = isMobile ? 'calc(100vw - 20px)' : '300px';
    
    errorDiv.style.cssText = `
        position: fixed;
        top: ${topPosition}px;
        right: ${rightPosition}px;
        background: #f44336;
        color: white;
        padding: ${isMobile ? '12px 16px' : '15px 20px'};
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(244, 67, 54, 0.3);
        z-index: 3000;
        display: flex;
        align-items: center;
        gap: 10px;
        animation: slideInRight 0.3s ease;
        max-width: ${maxWidth};
        font-size: ${isMobile ? '0.9rem' : '1rem'};
    `;

    document.body.appendChild(errorDiv);

    setTimeout(() => {
        errorDiv.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (document.body.contains(errorDiv)) {
                document.body.removeChild(errorDiv);
            }
        }, 300);
    }, 3000);
}

// Show Success Message
function showSuccessMessage() {
    const successDiv = document.createElement('div');
    successDiv.className = 'success-message';
    successDiv.innerHTML = `
        <i class="fas fa-check-circle"></i>
        <span>Pesanan Anda telah dikirim ke WhatsApp!</span>
    `;
    
    // Mobile responsive positioning
    const isMobile = window.innerWidth <= 768;
    const topPosition = isMobile ? 80 : 100;
    const rightPosition = isMobile ? 10 : 20;
    const maxWidth = isMobile ? 'calc(100vw - 20px)' : '300px';
    
    successDiv.style.cssText = `
        position: fixed;
        top: ${topPosition}px;
        right: ${rightPosition}px;
        background: #4caf50;
        color: white;
        padding: ${isMobile ? '12px 16px' : '15px 20px'};
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(76, 175, 80, 0.3);
        z-index: 3000;
        display: flex;
        align-items: center;
        gap: 10px;
        animation: slideInRight 0.3s ease;
        max-width: ${maxWidth};
        font-size: ${isMobile ? '0.9rem' : '1rem'};
    `;

    document.body.appendChild(successDiv);

    setTimeout(() => {
        successDiv.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (document.body.contains(successDiv)) {
                document.body.removeChild(successDiv);
            }
        }, 300);
    }, 4000);
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(100%);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// Image lazy loading
function setupLazyLoading() {
    const images = document.querySelectorAll('img[data-src]');
    
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.classList.remove('lazy');
                imageObserver.unobserve(img);
            }
        });
    });

    images.forEach(img => imageObserver.observe(img));
}

// Initialize lazy loading
setupLazyLoading();

// Add loading animation for images
document.addEventListener('DOMContentLoaded', function() {
    const images = document.querySelectorAll('.produk-image');
    images.forEach(img => {
        img.addEventListener('load', function() {
            this.style.animation = 'fadeIn 0.5s ease';
        });
        
        img.addEventListener('error', function() {
            this.src = 'https://via.placeholder.com/300x250/e91e63/ffffff?text=Queen+Florist';
        });
    });
});

// Performance optimization - Debounce scroll events
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Apply debounce to scroll events
window.addEventListener('scroll', debounce(function() {
    // Scroll-related functions here
}, 100));

// Add touch support for mobile
let touchStartX = 0;
let touchEndX = 0;

document.addEventListener('touchstart', function(e) {
    touchStartX = e.changedTouches[0].screenX;
});

document.addEventListener('touchend', function(e) {
    touchEndX = e.changedTouches[0].screenX;
    handleSwipe();
});

function handleSwipe() {
    if (touchEndX < touchStartX - 50) {
        // Swipe left - could be used for carousel navigation
        console.log('Swipe left');
    }
    if (touchEndX > touchStartX + 50) {
        // Swipe right - could be used for carousel navigation
        console.log('Swipe right');
    }
}

// Console branding
console.log('%c🌺 Queen Florist 🌺', 'font-size: 20px; color: #e91e63; font-weight: bold;');
console.log('%cTemukan Bunga Terindah untuk Setiap Momen', 'font-size: 14px; color: #9c27b0;');
console.log('%cCreated with ❤️ by brainmultimedia', 'font-size: 12px; color: #666;');