// Global Variables
let selectedProduct = null;
let currentAgenda = [];

// Data Produk
const productCategories = [
    {
        id: 1,
        name: "Standing Flower",
        icon: "fas fa-spa",
        image: "https://i.imgur.com/fRauqUO.jpeg",
        description: "Bunga standing elegan untuk berbagai acara"
    },
    {
        id: 2,
        name: "Paper Flower",
        icon: "fas fa-gift",
        image: "https://i.imgur.com/JqYHlaX.jpeg",
        description: "Bunga kertas cantik dan awet"
    },
    {
        id: 3,
        name: "Duka Cita",
        icon: "fas fa-heart",
        image: "https://i.imgur.com/JuJhAyP.jpeg",
        description: "Ungkapan simpati dan belasungkawa"
    },
    {
        id: 4,
        name: "Bouquet",
        icon: "fas fa-spa",
        image: "https://i.imgur.com/hl90R43.jpeg",
        description: "Buket bunga segar dan romantis"
    },
    {
        id: 5,
        name: "Wedding Car Decoration",
        icon: "fas fa-car",
        image: "https://i.imgur.com/zuueu4D.jpeg",
        description: "Dekorasi mobil pengantin mewah"
    },
    {
        id: 6,
        name: "Balloon Decoration",
        icon: "fas fa-circle",
        image: "https://i.imgur.com/mED89sx.jpeg",
        description: "Dekorasi balon meriah"
    },
    {
        id: 7,
        name: "Kado Rustic",
        icon: "fas fa-box",
        image: "https://i.imgur.com/Rja49Tr.jpeg",
        description: "Kado dengan gaya rustic unik"
    },
    {
        id: 8,
        name: "Balloon Parcel",
        icon: "fas fa-gift",
        image: "https://i.imgur.com/DPrCrrF.jpeg",
        description: "Parcel balon menarik"
    },
    {
        id: 9,
        name: "Snack Bouquet",
        icon: "fas fa-cookie",
        image: "https://i.imgur.com/QmJMTVR.jpeg",
        description: "Buket snack lezat"
    },
    {
        id: 10,
        name: "Tart Snack",
        icon: "fas fa-birthday-cake",
        image: "https://i.imgur.com/9gzsAw7.jpeg",
        description: "Tart dan snack lezat"
    }
];

// Data Agenda
const allAgendas = [
    { month: 9, day: 24, title: "Hari Dokter Indonesia", description: "Peringatan untuk tenaga medis Indonesia" },
    { month: 9, day: 27, title: "Hari Penerbangan Nasional", description: "Memperingati sejarah penerbangan Indonesia" },
    { month: 9, day: 28, title: "Hari Sumpah Pemuda", description: "Memperingati ikrar pemuda Indonesia 1928" },
    { month: 9, day: 30, title: "Hari Keuangan", description: "Hari kesadaran pentingnya literasi keuangan" },
    { month: 10, day: 1, title: "Hari Kesaktian Pancasila", description: "Memperingati jasa Pahlawan Revolusi" },
    { month: 10, day: 10, title: "Hari Kesehatan Jiwa", description: "Meningkatkan kesadaran kesehatan mental" },
    { month: 10, day: 16, title: "Hari Pangan Sedunia", description: "Meningkatkan kesadaran akan masalah pangan" },
    { month: 10, day: 24, title: "Hari Dokter Indonesia", description: "Menghormati jasa para dokter" },
    { month: 10, day: 28, title: "Hari Sumpah Pemuda", description: "Memperingati ikrar pemuda Indonesia" },
    { month: 10, day: 29, title: "Hari Stroke Nasional", description: "Meningkatkan kesadaran pencegahan stroke" },
    { month: 11, day: 10, title: "Hari Pahlawan", description: "Memperingati jasa pahlawan kemerdekaan" },
    { month: 11, day: 25, title: "Hari Guru Nasional", description: "Menghormati jasa para guru" }
];

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

// Initialize App
function initializeApp() {
    loadAgenda();
    loadProducts();
    setupEventListeners();
    setupSmoothScroll();
    setupScrollEffects();
}

// Setup Event Listeners
function setupEventListeners() {
    // Mobile menu toggle
    const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
    const navMenu = document.querySelector('.nav-menu');
    
    if (mobileMenuBtn && navMenu) {
        mobileMenuBtn.addEventListener('click', function() {
            navMenu.classList.toggle('active');
        });
    }

    // Close mobile menu when clicking on links
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('click', function() {
            navMenu.classList.remove('active');
        });
    });

    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
        const modal = document.getElementById('orderModal');
        if (event.target === modal) {
            closeOrderModal();
        }
    });

    // Form submission
    const orderForm = document.getElementById('orderForm');
    if (orderForm) {
        orderForm.addEventListener('submit', function(e) {
            e.preventDefault();
            submitOrder();
        });
    }
}

// Setup Smooth Scroll
function setupSmoothScroll() {
    const navLinks = document.querySelectorAll('.nav-link, .hero-buttons a[href^="#"]');
    
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href');
            if (targetId === '#') return;
            
            const targetSection = document.querySelector(targetId);
            if (targetSection) {
                const headerHeight = document.querySelector('.header').offsetHeight;
                const targetPosition = targetSection.offsetTop - headerHeight;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

// Setup Scroll Effects
function setupScrollEffects() {
    const header = document.querySelector('.header');
    
    window.addEventListener('scroll', function() {
        if (window.scrollY > 100) {
            header.style.background = 'rgba(255, 255, 255, 0.98)';
            header.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.1)';
        } else {
            header.style.background = 'rgba(255, 255, 255, 0.95)';
            header.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
        }
    });
}

// Show Loading
function showLoading() {
    const loadingElements = document.querySelectorAll('.loading');
    loadingElements.forEach(element => {
        element.style.display = 'flex';
    });
}

// Hide Loading
function hideLoading() {
    const loadingElements = document.querySelectorAll('.loading');
    loadingElements.forEach(element => {
        element.style.display = 'none';
    });
}

// Load Agenda from Database
async function loadAgenda() {
    const agendaGrid = document.getElementById('agendaGrid');
    if (!agendaGrid) return;

    try {
        showLoading();
        
        // Fetch from API
        const response = await fetch('./api/agenda.php');
        const result = await response.json();
        
        if (result.success) {
            const currentMonth = new Date().getMonth() + 1; // JavaScript months are 0-11
            currentAgenda = result.data.filter(agenda => agenda.month === currentMonth);

            if (currentAgenda.length === 0) {
                agendaGrid.innerHTML = `
                    <div class="no-agenda">
                        <i class="fas fa-calendar-alt agenda-icon"></i>
                        <p>Tidak ada agenda khusus bulan ini</p>
                    </div>
                `;
                return;
            }

            agendaGrid.innerHTML = currentAgenda.map(agenda => `
                <div class="agenda-card">
                    <i class="fas fa-calendar-alt agenda-icon"></i>
                    <div class="agenda-date">${agenda.day} ${getMonthName(agenda.month - 1)} ${new Date().getFullYear()}</div>
                    <h3 class="agenda-title">${agenda.title}</h3>
                    <p class="agenda-description">${agenda.description}</p>
                </div>
            `).join('');
        } else {
            throw new Error(result.message || 'Failed to load agenda');
        }
    } catch (error) {
        console.error('Error loading agenda:', error);
        
        // Fallback to static data if API fails
        const currentMonth = new Date().getMonth();
        currentAgenda = allAgendas.filter(agenda => agenda.month === currentMonth);

        if (currentAgenda.length === 0) {
            agendaGrid.innerHTML = `
                <div class="no-agenda">
                    <i class="fas fa-calendar-alt agenda-icon"></i>
                    <p>Tidak ada agenda khusus bulan ini</p>
                </div>
            `;
            return;
        }

        agendaGrid.innerHTML = currentAgenda.map(agenda => `
            <div class="agenda-card">
                <i class="fas fa-calendar-alt agenda-icon"></i>
                <div class="agenda-date">${agenda.day} ${getMonthName(currentMonth)} ${new Date().getFullYear()}</div>
                <h3 class="agenda-title">${agenda.title}</h3>
                <p class="agenda-description">${agenda.description}</p>
            </div>
        `).join('');
    } finally {
        hideLoading();
    }
}

// Load Products
function loadProducts() {
    const produkGrid = document.getElementById('produkGrid');
    if (!produkGrid) return;

    produkGrid.innerHTML = productCategories.map(product => `
        <div class="produk-card">
            <div class="produk-image">
                <img src="${product.image}" alt="${product.name}" loading="lazy">
            </div>
            <div class="produk-content">
                <i class="${product.icon} produk-icon"></i>
                <h3 class="produk-name">${product.name}</h3>
                <p class="produk-description">${product.description}</p>
                <button class="btn btn-whatsapp" onclick="openOrderModal(${product.id})">
                    <i class="fab fa-whatsapp"></i> Order Sekarang
                </button>
            </div>
        </div>
    `).join('');
}

// Get Month Name
function getMonthName(monthIndex) {
    const months = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
                   'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
    return months[monthIndex] || 'Unknown';
}

// WhatsApp Functions
function openWhatsAppChat() {
    const phoneNumber = '6281223968122';
    const message = 'Hai kak mimin yang baik hati, saya mau tanya-tanya produk Queen Florist...';
    const whatsappUrl = `https://wa.me/${phoneNumber}?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
}

function openWhatsAppOrder(message) {
    const phoneNumber = '6281223968122';
    const whatsappUrl = `https://wa.me/${phoneNumber}?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
}

// Order Modal Functions
function openOrderModal(productId) {
    selectedProduct = productCategories.find(p => p.id === productId);
    if (!selectedProduct) return;

    const modal = document.getElementById('orderModal');
    const modalImage = document.getElementById('modalProductImage');
    const modalName = document.getElementById('modalProductName');
    const modalDescription = document.getElementById('modalProductDescription');
    const modalLink = document.getElementById('modalProductLink');

    if (modal && modalImage && modalName && modalDescription && modalLink) {
        modalImage.src = selectedProduct.image;
        modalImage.alt = selectedProduct.name;
        modalName.textContent = selectedProduct.name;
        modalDescription.textContent = selectedProduct.description;
        modalLink.href = selectedProduct.image;
        modalLink.textContent = selectedProduct.image;

        modal.style.display = 'block';
        document.body.style.overflow = 'hidden';
    }
}

function closeOrderModal() {
    const modal = document.getElementById('orderModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
        
        // Reset form
        const orderForm = document.getElementById('orderForm');
        if (orderForm) {
            orderForm.reset();
        }
        
        selectedProduct = null;
    }
}

// Copy Link Function
function copyLink() {
    if (!selectedProduct) return;

    navigator.clipboard.writeText(selectedProduct.image).then(function() {
        showNotification('Link gambar berhasil disalin!');
    }).catch(function(err) {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = selectedProduct.image;
        document.body.appendChild(textArea);
        textArea.select();
        
        try {
            document.execCommand('copy');
            showNotification('Link gambar berhasil disalin!');
        } catch (err) {
            showNotification('Gagal menyalin link gambar');
        }
        
        document.body.removeChild(textArea);
    });
}

// Submit Order
function submitOrder() {
    if (!selectedProduct) {
        showNotification('Silakan pilih produk terlebih dahulu');
        return;
    }

    const customerName = document.getElementById('customerName').value.trim();
    const customerAddress = document.getElementById('customerAddress').value.trim();
    const customerNote = document.getElementById('customerNote').value.trim();

    // Validation
    if (!customerName) {
        showNotification('Mohon masukkan nama Anda');
        return;
    }

    if (!customerAddress) {
        showNotification('Mohon masukkan alamat pengiriman');
        return;
    }

    // Create WhatsApp message
    const message = `🌸 QUEEN FLORIST - PESANAN PRODUK 🌸

📸 *PRODUK YANG DIPESAN:*
🌷 Nama: ${selectedProduct.name}
📝 Deskripsi: ${selectedProduct.description}
🔗 Lihat Gambar: ${selectedProduct.image}

👤 *DATA PEMESAN:*
📝 Nama: ${customerName}
🏠 Alamat Kirim: ${customerAddress}
📋 Catatan: ${customerNote || 'Tidak ada'}

💬 *CARA PEMESANAN:*
1. Admin akan menghubungi Anda untuk konfirmasi
2. Pembayaran bisa dilakukan via transfer/ewallet
3. Pengiriman akan dijadwalkan setelah pembayaran

Mohon info lebih lanjut ya, terima kasih! 🙏

---
📱 Queen Florist Kebumen
📍 Gg. Kutilang No.7, Panjer, Kebumen
📞 +6281223968122`;

    // Send to WhatsApp
    openWhatsAppOrder(message);
    
    // Close modal after sending
    setTimeout(() => {
        closeOrderModal();
    }, 1000);
}

// Show Notification
function showNotification(message) {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = 'notification';
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #ec4899;
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 3000;
        font-size: 14px;
        font-weight: 500;
        animation: slideInRight 0.3s ease;
    `;

    // Add animation
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        @keyframes slideOutRight {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(100%);
                opacity: 0;
            }
        }
    `;
    document.head.appendChild(style);

    // Add to DOM
    document.body.appendChild(notification);

    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Utility Functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('id-ID', {
        style: 'currency',
        currency: 'IDR'
    }).format(amount);
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Image Lazy Loading
function setupLazyLoading() {
    const images = document.querySelectorAll('img[loading="lazy"]');
    
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src || img.src;
                    img.classList.remove('lazy');
                    imageObserver.unobserve(img);
                }
            });
        });

        images.forEach(img => imageObserver.observe(img));
    }
}

// Initialize lazy loading
document.addEventListener('DOMContentLoaded', setupLazyLoading);

// Error Handling
window.addEventListener('error', function(e) {
    console.error('JavaScript Error:', e.error);
    showNotification('Terjadi kesalahan. Silakan refresh halaman.');
});

// Performance Monitoring
window.addEventListener('load', function() {
    if (window.performance && window.performance.timing) {
        const loadTime = window.performance.timing.loadEventEnd - window.performance.timing.navigationStart;
        console.log('Page load time:', loadTime + 'ms');
    }
});

// Service Worker Registration (for PWA support)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        navigator.serviceWorker.register('/sw.js')
            .then(function(registration) {
                console.log('ServiceWorker registration successful');
            })
            .catch(function(error) {
                console.log('ServiceWorker registration failed');
            });
    });
}