/* =====================================================
   Agenda Management Panel - Ngadimin
   Production Ready JavaScript
   ===================================================== */

// Global Variables
let currentData = [];
let deleteId = null;
let currentConfigContent = '';

// DOM Elements
const scheduleTableBody = document.getElementById('scheduleTableBody');
const scheduleModal = document.getElementById('scheduleModal');
const deleteModal = document.getElementById('deleteModal');
const configModal = document.getElementById('configModal');
const scheduleForm = document.getElementById('scheduleForm');
const databaseConfigForm = document.getElementById('databaseConfigForm');
const loadingSpinner = document.getElementById('loadingSpinner');
const noDataMessage = document.getElementById('noDataMessage');
const searchInput = document.getElementById('searchInput');
const monthFilter = document.getElementById('monthFilter');
const statusFilter = document.getElementById('statusFilter');
const toast = document.getElementById('toast');
const toastMessage = document.getElementById('toastMessage');
const toastIcon = document.getElementById('toastIcon');

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    loadAgendaData();
    setupEventListeners();
});

// Setup Event Listeners
function setupEventListeners() {
    // Form submission
    scheduleForm.addEventListener('submit', handleFormSubmit);
    databaseConfigForm.addEventListener('submit', handleDatabaseConfigSubmit);
    
    // Search and filters
    searchInput.addEventListener('input', debounce(filterData, 300));
    monthFilter.addEventListener('change', filterData);
    statusFilter.addEventListener('change', filterData);
    
    // Status switch
    const isActiveSwitch = document.getElementById('isActive');
    const statusText = document.getElementById('statusText');
    isActiveSwitch.addEventListener('change', function() {
        statusText.textContent = this.checked ? 'Aktif' : 'Tidak Aktif';
    });
    
    // Close modals on outside click
    window.addEventListener('click', function(event) {
        if (event.target === scheduleModal) {
            closeModal();
        }
        if (event.target === deleteModal) {
            closeDeleteModal();
        }
        if (event.target === configModal) {
            closeConfigModal();
        }
    });
    
    // Escape key to close modals
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeModal();
            closeDeleteModal();
            closeConfigModal();
        }
    });
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(event) {
        // Ctrl/Cmd + N: New agenda
        if ((event.ctrlKey || event.metaKey) && event.key === 'n') {
            event.preventDefault();
            openAddModal();
        }
        
        // Ctrl/Cmd + F: Focus search
        if ((event.ctrlKey || event.metaKey) && event.key === 'f') {
            event.preventDefault();
            searchInput.focus();
        }
        
        // Ctrl/Cmd + E: Export
        if ((event.ctrlKey || event.metaKey) && event.key === 'e') {
            event.preventDefault();
            exportToCSV();
        }
        
        // Ctrl/Cmd + P: Print
        if ((event.ctrlKey || event.metaKey) && event.key === 'p') {
            event.preventDefault();
            printData();
        }
    });
}

// API Functions
async function apiRequest(endpoint, method = 'GET', data = null) {
    try {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
            }
        };
        
        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }
        
        const response = await fetch(`api/${endpoint}`, options);
        
        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        
        return await response.json();
    } catch (error) {
        console.error('API Error:', error);
        showToast('Terjadi kesalahan saat menghubungi server: ' + error.message, 'error');
        throw error;
    }
}

// Load Agenda Data
async function loadAgendaData() {
    showLoading(true);
    try {
        const response = await apiRequest('agenda.php');
        currentData = response.data || [];
        renderTable(currentData);
        updateStats();
    } catch (error) {
        console.error('Error loading data:', error);
        showToast('Gagal memuat data', 'error');
    } finally {
        showLoading(false);
    }
}

// Render Table
function renderTable(data) {
    if (data.length === 0) {
        scheduleTableBody.innerHTML = '';
        noDataMessage.style.display = 'block';
        return;
    }
    
    noDataMessage.style.display = 'none';
    
    scheduleTableBody.innerHTML = data.map(item => `
        <tr>
            <td>${item.id}</td>
            <td>${formatDate(item.month, item.day)}</td>
            <td><strong>${escapeHtml(item.title)}</strong></td>
            <td>${escapeHtml(item.description)}</td>
            <td>
                <span class="status-badge ${item.is_active ? 'status-active' : 'status-inactive'}">
                    ${item.is_active ? 'Aktif' : 'Tidak Aktif'}
                </span>
            </td>
            <td>
                <div class="action-buttons">
                    <button class="btn btn-sm btn-warning" onclick="editAgenda(${item.id})" title="Edit">
                        <i class="fas fa-edit"></i> Edit
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteAgenda(${item.id})" title="Hapus">
                        <i class="fas fa-trash"></i> Hapus
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Format Date
function formatDate(month, day) {
    const months = [
        'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
        'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
    ];
    return `${day} ${months[month - 1]}`;
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Update Statistics
function updateStats() {
    const totalEvents = currentData.length;
    const activeEvents = currentData.filter(item => item.is_active).length;
    const today = new Date();
    const upcomingEvents = currentData.filter(item => {
        const eventDate = new Date(today.getFullYear(), item.month - 1, item.day);
        return eventDate >= today;
    }).length;
    
    document.getElementById('totalEvents').textContent = totalEvents;
    document.getElementById('activeEvents').textContent = activeEvents;
    document.getElementById('upcomingEvents').textContent = upcomingEvents;
}

// Filter Data
function filterData() {
    const searchTerm = searchInput.value.toLowerCase();
    const selectedMonth = monthFilter.value;
    const selectedStatus = statusFilter.value;
    
    const filteredData = currentData.filter(item => {
        const matchesSearch = item.title.toLowerCase().includes(searchTerm) || 
                             item.description.toLowerCase().includes(searchTerm);
        const matchesMonth = !selectedMonth || item.month == selectedMonth;
        const matchesStatus = selectedStatus === '' || item.is_active == selectedStatus;
        
        return matchesSearch && matchesMonth && matchesStatus;
    });
    
    renderTable(filteredData);
}

// Modal Functions
function openAddModal() {
    document.getElementById('modalTitle').textContent = 'Tambah Agenda Baru';
    scheduleForm.reset();
    document.getElementById('scheduleId').value = '';
    document.getElementById('statusText').textContent = 'Tidak Aktif';
    scheduleModal.classList.add('show');
    document.getElementById('title').focus();
}

function openEditModal(item) {
    document.getElementById('modalTitle').textContent = 'Edit Agenda';
    document.getElementById('scheduleId').value = item.id;
    document.getElementById('month').value = item.month;
    document.getElementById('day').value = item.day;
    document.getElementById('title').value = item.title;
    document.getElementById('description').value = item.description;
    document.getElementById('isActive').checked = item.is_active;
    document.getElementById('statusText').textContent = item.is_active ? 'Aktif' : 'Tidak Aktif';
    scheduleModal.classList.add('show');
    document.getElementById('title').focus();
}

function closeModal() {
    scheduleModal.classList.remove('show');
    scheduleForm.reset();
}

function closeDeleteModal() {
    deleteModal.classList.remove('show');
    deleteId = null;
}

// CRUD Operations
async function handleFormSubmit(e) {
    e.preventDefault();
    
    if (!validateForm()) {
        return;
    }
    
    const formData = {
        month: parseInt(document.getElementById('month').value),
        day: parseInt(document.getElementById('day').value),
        title: document.getElementById('title').value.trim(),
        description: document.getElementById('description').value.trim(),
        is_active: document.getElementById('isActive').checked ? 1 : 0
    };
    
    const id = document.getElementById('scheduleId').value;
    const isEdit = id !== '';
    
    try {
        showLoading(true);
        
        if (isEdit) {
            await apiRequest(`agenda.php?id=${id}`, 'PUT', formData);
            showToast('Agenda berhasil diperbarui', 'success');
        } else {
            await apiRequest('agenda.php', 'POST', formData);
            showToast('Agenda berhasil ditambahkan', 'success');
        }
        
        closeModal();
        await loadAgendaData();
    } catch (error) {
        console.error('Error saving data:', error);
        showToast(isEdit ? 'Gagal memperbarui agenda' : 'Gagal menambahkan agenda', 'error');
    } finally {
        showLoading(false);
    }
}

async function editAgenda(id) {
    const item = currentData.find(agenda => agenda.id === id);
    if (item) {
        openEditModal(item);
    } else {
        showToast('Agenda tidak ditemukan', 'error');
    }
}

function deleteAgenda(id) {
    deleteId = id;
    deleteModal.classList.add('show');
}

async function confirmDelete() {
    if (!deleteId) return;
    
    try {
        showLoading(true);
        await apiRequest(`agenda.php?id=${deleteId}`, 'DELETE');
        showToast('Agenda berhasil dihapus', 'success');
        closeDeleteModal();
        await loadAgendaData();
    } catch (error) {
        console.error('Error deleting data:', error);
        showToast('Gagal menghapus agenda', 'error');
    } finally {
        showLoading(false);
    }
}

// Loading State
function showLoading(show) {
    loadingSpinner.style.display = show ? 'flex' : 'none';
}

// Toast Notification
function showToast(message, type = 'success') {
    toastMessage.textContent = message;
    toast.className = `toast ${type}`;
    
    // Set icon based on type
    const iconClasses = {
        success: 'fas fa-check-circle',
        error: 'fas fa-exclamation-circle',
        warning: 'fas fa-exclamation-triangle'
    };
    
    toastIcon.className = iconClasses[type] || iconClasses.success;
    
    // Show toast
    setTimeout(() => {
        toast.classList.add('show');
    }, 100);
    
    // Hide toast after 3 seconds
    setTimeout(() => {
        toast.classList.remove('show');
    }, 3000);
}

// Form Validation
function validateForm() {
    const month = document.getElementById('month').value;
    const day = document.getElementById('day').value;
    const title = document.getElementById('title').value.trim();
    const description = document.getElementById('description').value.trim();
    
    if (!month) {
        showToast('Silakan pilih bulan', 'warning');
        document.getElementById('month').focus();
        return false;
    }
    
    if (!day || day < 1 || day > 31) {
        showToast('Silakan masukkan hari yang valid (1-31)', 'warning');
        document.getElementById('day').focus();
        return false;
    }
    
    if (!title) {
        showToast('Silakan masukkan judul', 'warning');
        document.getElementById('title').focus();
        return false;
    }
    
    if (!description) {
        showToast('Silakan masukkan deskripsi', 'warning');
        document.getElementById('description').focus();
        return false;
    }
    
    // Validate day based on month
    const daysInMonth = new Date(2024, month, 0).getDate();
    if (day > daysInMonth) {
        showToast(`Hari tidak valid untuk bulan yang dipilih (maksimal ${daysInMonth} hari)`, 'warning');
        document.getElementById('day').focus();
        return false;
    }
    
    return true;
}

// Refresh Data
async function refreshData() {
    await loadAgendaData();
    showToast('Data berhasil diperbarui', 'success');
}

// Export Functions
function exportToCSV() {
    if (currentData.length === 0) {
        showToast('Tidak ada data untuk diekspor', 'warning');
        return;
    }
    
    const headers = ['ID', 'Bulan', 'Hari', 'Judul', 'Deskripsi', 'Status'];
    const rows = currentData.map(item => [
        item.id,
        item.month,
        item.day,
        `"${item.title}"`,
        `"${item.description}"`,
        item.is_active ? 'Aktif' : 'Tidak Aktif'
    ]);
    
    const csvContent = [headers, ...rows]
        .map(row => row.join(','))
        .join('\n');
    
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `agenda_${new Date().toISOString().split('T')[0]}.csv`;
    a.click();
    window.URL.revokeObjectURL(url);
    
    showToast('Data berhasil diekspor ke CSV', 'success');
}

function printData() {
    window.print();
}

// Utility Functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Auto-refresh (optional - every 5 minutes)
setInterval(async function() {
    try {
        const response = await apiRequest('agenda.php');
        const newData = response.data || [];
        
        // Only update if data has changed
        if (JSON.stringify(newData) !== JSON.stringify(currentData)) {
            currentData = newData;
            renderTable(currentData);
            updateStats();
            showToast('Agenda diperbarui', 'success');
        }
    } catch (error) {
        console.error('Auto-refresh error:', error);
    }
}, 300000); // 5 minutes

// Service Worker Registration (for PWA support - optional)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        navigator.serviceWorker.register('/sw.js')
            .then(function(registration) {
                console.log('ServiceWorker registration successful with scope: ', registration.scope);
            })
            .catch(function(err) {
                console.log('ServiceWorker registration failed: ', err);
            });
    });
}

// Error Handling
window.addEventListener('error', function(event) {
    console.error('Global error:', event.error);
    showToast('Terjadi kesalahan yang tidak terduga', 'error');
});

window.addEventListener('unhandledrejection', function(event) {
    console.error('Unhandled promise rejection:', event.reason);
    showToast('Terjadi kesalahan yang tidak terduga', 'error');
});

// Performance Monitoring
if (window.performance && window.performance.navigation) {
    window.addEventListener('load', function() {
        const perfData = window.performance.timing;
        const pageLoadTime = perfData.loadEventEnd - perfData.navigationStart;
        console.log('Page load time:', pageLoadTime + 'ms');
    });
}

// Configuration Management Functions
function openConfigModal() {
    configModal.classList.add('show');
    loadConfigFile();
}

function closeConfigModal() {
    configModal.classList.remove('show');
}

function switchConfigTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.config-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById(tabName + 'Tab').classList.add('active');
    
    // Add active class to clicked button
    event.target.classList.add('active');
    
    // Load config file when switching to file tab
    if (tabName === 'file') {
        loadConfigFile();
    }
}

async function loadConfigFile() {
    try {
        const response = await fetch('api/config.php');
        const content = await response.text();
        document.getElementById('configContent').value = content;
        currentConfigContent = content;
    } catch (error) {
        console.error('Error loading config file:', error);
        showToast('Gagal memuat file konfigurasi', 'error');
    }
}

async function saveConfigFile() {
    const filepath = document.getElementById('configFile').value;
    const content = document.getElementById('configContent').value;
    
    if (!filepath || !content) {
        showToast('Filepath dan konten tidak boleh kosong', 'warning');
        return;
    }
    
    try {
        showLoading(true);
        
        const response = await fetch('api/config_update.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                filepath: filepath,
                content: content
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast('Konfigurasi berhasil disimpan', 'success');
            currentConfigContent = content;
        } else {
            showToast('Gagal menyimpan konfigurasi: ' + result.message, 'error');
        }
    } catch (error) {
        console.error('Error saving config file:', error);
        showToast('Terjadi kesalahan saat menyimpan konfigurasi', 'error');
    } finally {
        showLoading(false);
    }
}

function validateConfigContent() {
    const content = document.getElementById('configContent').value;
    
    if (!content.trim()) {
        showToast('Konten konfigurasi kosong', 'warning');
        return;
    }
    
    // Basic PHP syntax validation
    if (content.includes('<?php')) {
        // Check for basic PHP syntax errors
        const openTags = (content.match(/<\?php/g) || []).length;
        const closeTags = (content.match(/\?>/g) || []).length;
        
        if (openTags !== closeTags) {
            showToast('Peringatan: Jumlah tag PHP tidak seimbang', 'warning');
            return;
        }
        
        // Check for common syntax errors
        if (content.includes(';;') || content.includes(',,') || content.includes('{{')) {
            showToast('Peringatan: Terdeteksi kemungkinan kesalahan sintaks', 'warning');
            return;
        }
    }
    
    showToast('Validasi sintaks berhasil', 'success');
}

async function handleDatabaseConfigSubmit(e) {
    e.preventDefault();
    
    const formData = {
        db_host: document.getElementById('dbHost').value,
        db_name: document.getElementById('dbName').value,
        db_user: document.getElementById('dbUser').value,
        db_pass: document.getElementById('dbPass').value
    };
    
    try {
        showLoading(true);
        
        // Generate new config content
        const newConfigContent = generateConfigContent(formData);
        
        // Update the file content textarea
        document.getElementById('configContent').value = newConfigContent;
        
        // Switch to file tab
        switchConfigTab('file');
        
        showToast('Konfigurasi database berhasil dibuat. Silakan simpan file.', 'success');
        
    } catch (error) {
        console.error('Error generating database config:', error);
        showToast('Gagal membuat konfigurasi database', 'error');
    } finally {
        showLoading(false);
    }
}

function generateConfigContent(formData) {
    return `<?php
/* =====================================================
   Database Configuration - Ngadimin Panel
   Production Ready Configuration
   ===================================================== */

// Database Configuration - Update these values for your hosting
define('DB_HOST', '${formData.db_host}');
define('DB_NAME', '${formData.db_name}');
define('DB_USER', '${formData.db_user}');
define('DB_PASS', '${formData.db_pass}');

// CORS Headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Content Type
header('Content-Type: application/json');

// Error Reporting (Disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Set to 0 in production
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/error.log');

// Database Connection
function getConnection() {
    try {
        $conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4", DB_USER, DB_PASS);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        return $conn;
    } catch(PDOException $e) {
        // Log error
        error_log("Database connection failed: " . $e->getMessage());
        
        // Return generic error message
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Database connection failed. Please check your configuration.'
        ]);
        exit();
    }
}

// Helper Functions
function sendResponse($success, $message, $data = null, $statusCode = 200) {
    http_response_code($statusCode);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    exit();
}

function getInputData() {
    $input = file_get_contents('php://input');
    if (empty($input)) {
        return null;
    }
    
    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("JSON decode error: " . json_last_error_msg());
        return null;
    }
    
    return $data;
}

function validateRequired($data, $requiredFields) {
    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || $data[$field] === '' || $data[$field] === null) {
            return false;
        }
    }
    return true;
}

function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

// Additional helper functions for agenda.php
function validateMonth($month) {
    return is_numeric($month) && $month >= 1 && $month <= 12;
}

function validateDay($day, $month) {
    if (!is_numeric($day) || $day < 1 || $day > 31) {
        return false;
    }
    
    // Check day based on month
    $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, 2024);
    return $day <= $daysInMonth;
}

function validateTitle($title) {
    return !empty(trim($title)) && strlen($title) <= 255;
}

function validateDescription($description) {
    return !empty(trim($description));
}

// Security Headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Rate Limiting (Simple implementation)
function checkRateLimit() {
    $clientIp = $_SERVER['REMOTE_ADDR'];
    $currentTime = time();
    $window = 60; // 1 minute window
    $maxRequests = 100; // Max 100 requests per minute
    
    // Simple file-based rate limiting
    $rateLimitFile = __DIR__ . '/../logs/rate_limit_' . md5($clientIp) . '.json';
    
    if (file_exists($rateLimitFile)) {
        $data = json_decode(file_get_contents($rateLimitFile), true);
        
        // Reset if window expired
        if ($currentTime - $data['reset_time'] > $window) {
            $data = ['count' => 0, 'reset_time' => $currentTime];
        }
        
        // Check limit
        if ($data['count'] >= $maxRequests) {
            http_response_code(429);
            echo json_encode([
                'success' => false,
                'message' => 'Too many requests. Please try again later.'
            ]);
            exit();
        }
        
        $data['count']++;
    } else {
        $data = ['count' => 1, 'reset_time' => $currentTime];
    }
    
    file_put_contents($rateLimitFile, json_encode($data));
}

// Apply rate limiting to POST, PUT, DELETE requests
if (in_array($_SERVER['REQUEST_METHOD'], ['POST', 'PUT', 'DELETE'])) {
    checkRateLimit();
}

// Create logs directory if it doesn't exist
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}
?>`;
}

async function testDatabaseConnection() {
    const dbHost = document.getElementById('dbHost').value;
    const dbName = document.getElementById('dbName').value;
    const dbUser = document.getElementById('dbUser').value;
    const dbPass = document.getElementById('dbPass').value;
    
    if (!dbHost || !dbName || !dbUser) {
        showToast('Host, nama database, dan user harus diisi', 'warning');
        return;
    }
    
    try {
        showLoading(true);
        
        // Create a test connection script
        const testScript = `<?php
try {
    $conn = new PDO("mysql:host=${dbHost};dbname=${dbName};charset=utf8mb4", "${dbUser}", "${dbPass}");
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo json_encode(['success' => true, 'message' => 'Koneksi database berhasil']);
} catch(PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'Koneksi gagal: ' . $e->getMessage()]);
}
?>`;
        
        // For security reasons, we'll just validate the input format
        if (dbHost && dbName && dbUser) {
            showToast('Format konfigurasi valid. Silakan simpan dan test koneksi langsung.', 'success');
        } else {
            showToast('Format konfigurasi tidak valid', 'error');
        }
        
    } catch (error) {
        console.error('Error testing database connection:', error);
        showToast('Terjadi kesalahan saat test koneksi', 'error');
    } finally {
        showLoading(false);
    }
}