<?php
/* =====================================================
   Configuration Update API - Ngadimin Panel
   Handles configuration file updates with proper validation
   ===================================================== */

// Include configuration
require_once 'config.php';

// Security check - Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Method not allowed', null, 405);
}

// Get input data
$inputData = getInputData();

// Validate required fields
$requiredFields = ['filepath', 'content'];
if (!validateRequired($inputData, $requiredFields)) {
    sendResponse(false, 'Input validation error: Missing required fields (filepath, content)', null, 400);
}

// Extract and validate filepath
$filepath = $inputData['filepath'];
$content = $inputData['content'];

// Security: Validate filepath to prevent directory traversal
$allowedPaths = [
    __DIR__ . '/config.php',
    __DIR__ . '/../config.php',
    '/home/z/my-project/neoqueen/ngadimin/deploy/api/config.php'
];

$realPath = realpath(dirname($filepath) . '/' . basename($filepath));
$allowedRealPaths = array_map('realpath', $allowedPaths);

if (!in_array($realPath, $allowedRealPaths)) {
    sendResponse(false, 'Security error: Filepath not allowed', null, 403);
}

// Check if file exists and is writable
if (!file_exists($filepath)) {
    sendResponse(false, 'File not found: ' . $filepath, null, 404);
}

if (!is_writable($filepath)) {
    sendResponse(false, 'File not writable: ' . $filepath, null, 403);
}

// Validate content (basic PHP syntax check for .php files)
if (pathinfo($filepath, PATHINFO_EXTENSION) === 'php') {
    // Create temporary file to validate PHP syntax
    $tempFile = tempnam(sys_get_temp_dir(), 'php_config_check_');
    file_put_contents($tempFile, $content);
    
    // Check PHP syntax
    $output = [];
    $returnCode = 0;
    exec("php -l " . escapeshellarg($tempFile) . " 2>&1", $output, $returnCode);
    
    // Clean up temp file
    unlink($tempFile);
    
    if ($returnCode !== 0) {
        sendResponse(false, 'PHP syntax error: ' . implode(' ', $output), null, 400);
    }
}

// Create backup before updating
$backupFile = $filepath . '.backup.' . date('Y-m-d_H-i-s');
if (!copy($filepath, $backupFile)) {
    error_log("Failed to create backup of config file: " . $filepath);
}

// Update the file
try {
    $result = file_put_contents($filepath, $content);
    
    if ($result === false) {
        // Restore from backup if update failed
        if (file_exists($backupFile)) {
            copy($backupFile, $filepath);
        }
        sendResponse(false, 'Failed to update configuration file', null, 500);
    }
    
    // Log the update
    error_log("Configuration file updated: " . $filepath . " by " . $_SERVER['REMOTE_ADDR']);
    
    sendResponse(true, 'Configuration updated successfully', [
        'filepath' => $filepath,
        'backup_file' => $backupFile,
        'updated_at' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Restore from backup if exception occurred
    if (file_exists($backupFile)) {
        copy($backupFile, $filepath);
    }
    
    error_log("Configuration update error: " . $e->getMessage());
    sendResponse(false, 'Error updating configuration: ' . $e->getMessage(), null, 500);
}
?>