<?php
/* =====================================================
   Database Configuration - Ngadimin Panel
   Production Ready Configuration
   ===================================================== */

// Database Configuration - Update these values for your hosting
define('DB_HOST', 'localhost');
define('DB_NAME', 'queenflo_agenda');
define('DB_USER', 'root');
define('DB_PASS', 'pwtkbm30S!');

// CORS Headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Content Type
header('Content-Type: application/json');

// Error Reporting (Disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Set to 0 in production
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/error.log');

// Database Connection
function getConnection() {
    try {
        $conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4", DB_USER, DB_PASS);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        return $conn;
    } catch(PDOException $e) {
        // Log error
        error_log("Database connection failed: " . $e->getMessage());
        
        // Return generic error message
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Database connection failed. Please check your configuration.'
        ]);
        exit();
    }
}

// Helper Functions
function sendResponse($success, $message, $data = null, $statusCode = 200) {
    http_response_code($statusCode);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    exit();
}

function getInputData() {
    $input = file_get_contents('php://input');
    if (empty($input)) {
        return null;
    }
    
    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("JSON decode error: " . json_last_error_msg());
        return null;
    }
    
    return $data;
}

function validateRequired($data, $requiredFields) {
    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || $data[$field] === '' || $data[$field] === null) {
            return false;
        }
    }
    return true;
}

function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

// Additional helper functions for agenda.php
function validateMonth($month) {
    return is_numeric($month) && $month >= 1 && $month <= 12;
}

function validateDay($day, $month) {
    if (!is_numeric($day) || $day < 1 || $day > 31) {
        return false;
    }
    
    // Check day based on month
    $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, 2024);
    return $day <= $daysInMonth;
}

function validateTitle($title) {
    return !empty(trim($title)) && strlen($title) <= 255;
}

function validateDescription($description) {
    return !empty(trim($description));
}

// Security Headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Rate Limiting (Simple implementation)
function checkRateLimit() {
    $clientIp = $_SERVER['REMOTE_ADDR'];
    $currentTime = time();
    $window = 60; // 1 minute window
    $maxRequests = 100; // Max 100 requests per minute
    
    // Simple file-based rate limiting
    $rateLimitFile = __DIR__ . '/../logs/rate_limit_' . md5($clientIp) . '.json';
    
    if (file_exists($rateLimitFile)) {
        $data = json_decode(file_get_contents($rateLimitFile), true);
        
        // Reset if window expired
        if ($currentTime - $data['reset_time'] > $window) {
            $data = ['count' => 0, 'reset_time' => $currentTime];
        }
        
        // Check limit
        if ($data['count'] >= $maxRequests) {
            http_response_code(429);
            echo json_encode([
                'success' => false,
                'message' => 'Too many requests. Please try again later.'
            ]);
            exit();
        }
        
        $data['count']++;
    } else {
        $data = ['count' => 1, 'reset_time' => $currentTime];
    }
    
    file_put_contents($rateLimitFile, json_encode($data));
}

// Apply rate limiting to POST, PUT, DELETE requests
if (in_array($_SERVER['REQUEST_METHOD'], ['POST', 'PUT', 'DELETE'])) {
    checkRateLimit();
}

// Create logs directory if it doesn't exist
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}
?>