<?php
require_once 'config.php';

$database = new Database();
$conn = $database->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_SERVER['REQUEST_URI'];

try {
    switch ($method) {
        case 'GET':
            handleGet($conn);
            break;
        case 'POST':
            handlePost($conn);
            break;
        case 'PUT':
            handlePut($conn);
            break;
        case 'DELETE':
            handleDelete($conn);
            break;
        default:
            sendError('Method not allowed', 405);
    }
} catch (Exception $e) {
    logApiCall($endpoint, $method, null, 500);
    sendError('Internal server error: ' . $e->getMessage(), 500);
}

function handleGet($conn) {
    global $endpoint;
    
    // Get query parameters
    $month = isset($_GET['month']) ? (int)$_GET['month'] : null;
    $active = isset($_GET['active']) ? (int)$_GET['active'] : null;
    $search = isset($_GET['search']) ? $_GET['search'] : null;
    
    $sql = "SELECT * FROM agenda WHERE 1=1";
    $params = [];
    $types = "";
    
    if ($month !== null) {
        $sql .= " AND month = ?";
        $params[] = $month;
        $types .= "i";
    }
    
    if ($active !== null) {
        $sql .= " AND is_active = ?";
        $params[] = $active;
        $types .= "i";
    }
    
    if ($search) {
        $sql .= " AND (title LIKE ? OR description LIKE ?)";
        $searchParam = "%" . $conn->real_escape_string($search) . "%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $types .= "ss";
    }
    
    $sql .= " ORDER BY month, day";
    
    $stmt = $conn->prepare($sql);
    
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $agendas = [];
    while ($row = $result->fetch_assoc()) {
        $agendas[] = [
            'id' => (int)$row['id'],
            'month' => (int)$row['month'],
            'day' => (int)$row['day'],
            'title' => $row['title'],
            'description' => $row['description'],
            'is_active' => (int)$row['is_active'],
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at']
        ];
    }
    
    $stmt->close();
    
    logApiCall($endpoint, 'GET', $_GET, 200);
    sendResponse([
        'success' => true,
        'data' => $agendas,
        'total' => count($agendas)
    ]);
}

function handlePost($conn) {
    global $endpoint;
    
    validateToken();
    
    $data = getRequestBody();
    
    // Validate required fields
    if (!isset($data['month']) || !isset($data['day']) || !isset($data['title'])) {
        sendError('Missing required fields: month, day, title');
    }
    
    $month = (int)$data['month'];
    $day = (int)$data['day'];
    $title = trim($data['title']);
    $description = isset($data['description']) ? trim($data['description']) : '';
    $isActive = isset($data['is_active']) ? (int)$data['is_active'] : 1;
    
    // Validate values
    if ($month < 1 || $month > 12) {
        sendError('Month must be between 1 and 12');
    }
    
    if ($day < 1 || $day > 31) {
        sendError('Day must be between 1 and 31');
    }
    
    if (empty($title)) {
        sendError('Title cannot be empty');
    }
    
    // Check for duplicate
    $checkStmt = $conn->prepare("SELECT id FROM agenda WHERE month = ? AND day = ?");
    $checkStmt->bind_param("ii", $month, $day);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult->num_rows > 0) {
        sendError('Agenda for this date already exists');
    }
    $checkStmt->close();
    
    // Insert new agenda
    $stmt = $conn->prepare("
        INSERT INTO agenda (month, day, title, description, is_active) 
        VALUES (?, ?, ?, ?, ?)
    ");
    
    $stmt->bind_param("iissi", $month, $day, $title, $description, $isActive);
    
    if ($stmt->execute()) {
        $id = $conn->insert_id;
        
        $newAgenda = [
            'id' => $id,
            'month' => $month,
            'day' => $day,
            'title' => $title,
            'description' => $description,
            'is_active' => $isActive,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $stmt->close();
        
        logApiCall($endpoint, 'POST', $data, 201);
        sendResponse([
            'success' => true,
            'message' => 'Agenda created successfully',
            'data' => $newAgenda
        ], 201);
    } else {
        $stmt->close();
        sendError('Failed to create agenda');
    }
}

function handlePut($conn) {
    global $endpoint;
    
    validateToken();
    
    $data = getRequestBody();
    
    if (!isset($data['id'])) {
        sendError('Missing agenda ID');
    }
    
    $id = (int)$data['id'];
    
    // Check if agenda exists
    $checkStmt = $conn->prepare("SELECT id FROM agenda WHERE id = ?");
    $checkStmt->bind_param("i", $id);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult->num_rows === 0) {
        $checkStmt->close();
        sendError('Agenda not found', 404);
    }
    $checkStmt->close();
    
    // Validate and prepare update data
    $updateFields = [];
    $params = [];
    $types = "";
    
    if (isset($data['month'])) {
        $month = (int)$data['month'];
        if ($month < 1 || $month > 12) {
            sendError('Month must be between 1 and 12');
        }
        $updateFields[] = "month = ?";
        $params[] = $month;
        $types .= "i";
    }
    
    if (isset($data['day'])) {
        $day = (int)$data['day'];
        if ($day < 1 || $day > 31) {
            sendError('Day must be between 1 and 31');
        }
        $updateFields[] = "day = ?";
        $params[] = $day;
        $types .= "i";
    }
    
    if (isset($data['title'])) {
        $title = trim($data['title']);
        if (empty($title)) {
            sendError('Title cannot be empty');
        }
        $updateFields[] = "title = ?";
        $params[] = $title;
        $types .= "s";
    }
    
    if (isset($data['description'])) {
        $description = trim($data['description']);
        $updateFields[] = "description = ?";
        $params[] = $description;
        $types .= "s";
    }
    
    if (isset($data['is_active'])) {
        $isActive = (int)$data['is_active'];
        $updateFields[] = "is_active = ?";
        $params[] = $isActive;
        $types .= "i";
    }
    
    if (empty($updateFields)) {
        sendError('No fields to update');
    }
    
    // Check for duplicate (if month and day are being updated)
    if (isset($data['month']) && isset($data['day'])) {
        $duplicateCheck = $conn->prepare("SELECT id FROM agenda WHERE month = ? AND day = ? AND id != ?");
        $duplicateCheck->bind_param("iii", $data['month'], $data['day'], $id);
        $duplicateCheck->execute();
        $duplicateResult = $duplicateCheck->get_result();
        
        if ($duplicateResult->num_rows > 0) {
            $duplicateCheck->close();
            sendError('Agenda for this date already exists');
        }
        $duplicateCheck->close();
    }
    
    // Add updated_at field
    $updateFields[] = "updated_at = NOW()";
    
    // Add ID to params for WHERE clause
    $params[] = $id;
    $types .= "i";
    
    // Build and execute update query
    $sql = "UPDATE agenda SET " . implode(", ", $updateFields) . " WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    
    if ($stmt->execute()) {
        $stmt->close();
        
        // Get updated agenda
        $selectStmt = $conn->prepare("SELECT * FROM agenda WHERE id = ?");
        $selectStmt->bind_param("i", $id);
        $selectStmt->execute();
        $result = $selectStmt->get_result();
        $updatedAgenda = $result->fetch_assoc();
        $selectStmt->close();
        
        logApiCall($endpoint, 'PUT', $data, 200);
        sendResponse([
            'success' => true,
            'message' => 'Agenda updated successfully',
            'data' => [
                'id' => (int)$updatedAgenda['id'],
                'month' => (int)$updatedAgenda['month'],
                'day' => (int)$updatedAgenda['day'],
                'title' => $updatedAgenda['title'],
                'description' => $updatedAgenda['description'],
                'is_active' => (int)$updatedAgenda['is_active'],
                'created_at' => $updatedAgenda['created_at'],
                'updated_at' => $updatedAgenda['updated_at']
            ]
        ]);
    } else {
        $stmt->close();
        sendError('Failed to update agenda');
    }
}

function handleDelete($conn) {
    global $endpoint;
    
    validateToken();
    
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        sendError('Missing agenda ID');
    }
    
    // Check if agenda exists
    $checkStmt = $conn->prepare("SELECT title FROM agenda WHERE id = ?");
    $checkStmt->bind_param("i", $id);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    
    if ($checkResult->num_rows === 0) {
        $checkStmt->close();
        sendError('Agenda not found', 404);
    }
    
    $agenda = $checkResult->fetch_assoc();
    $checkStmt->close();
    
    // Delete agenda
    $stmt = $conn->prepare("DELETE FROM agenda WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        $stmt->close();
        
        logApiCall($endpoint, 'DELETE', ['id' => $id], 200);
        sendResponse([
            'success' => true,
            'message' => 'Agenda deleted successfully',
            'data' => [
                'id' => $id,
                'title' => $agenda['title']
            ]
        ]);
    } else {
        $stmt->close();
        sendError('Failed to delete agenda');
    }
}

$conn->close();
?>