// Global Variables
let currentData = [];
let deleteId = null;

// DOM Elements
const scheduleTableBody = document.getElementById('scheduleTableBody');
const scheduleModal = document.getElementById('scheduleModal');
const deleteModal = document.getElementById('deleteModal');
const scheduleForm = document.getElementById('scheduleForm');
const loadingSpinner = document.getElementById('loadingSpinner');
const noDataMessage = document.getElementById('noDataMessage');
const searchInput = document.getElementById('searchInput');
const monthFilter = document.getElementById('monthFilter');
const statusFilter = document.getElementById('statusFilter');
const toast = document.getElementById('toast');
const toastMessage = document.getElementById('toastMessage');
const toastIcon = document.getElementById('toastIcon');

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    loadAgendaData();
    setupEventListeners();
});

// Setup Event Listeners
function setupEventListeners() {
    // Form submission
    scheduleForm.addEventListener('submit', handleFormSubmit);
    
    // Search and filters
    searchInput.addEventListener('input', filterData);
    monthFilter.addEventListener('change', filterData);
    statusFilter.addEventListener('change', filterData);
    
    // Status switch
    const isActiveSwitch = document.getElementById('isActive');
    const statusText = document.getElementById('statusText');
    isActiveSwitch.addEventListener('change', function() {
        statusText.textContent = this.checked ? 'Aktif' : 'Tidak Aktif';
    });
    
    // Close modals on outside click
    window.addEventListener('click', function(event) {
        if (event.target === scheduleModal) {
            closeModal();
        }
        if (event.target === deleteModal) {
            closeDeleteModal();
        }
    });
    
    // Escape key to close modals
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeModal();
            closeDeleteModal();
        }
    });
}

// API Functions
async function apiRequest(endpoint, method = 'GET', data = null) {
    try {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
            }
        };
        
        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }
        
        const response = await fetch(`api/${endpoint}`, options);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    } catch (error) {
        console.error('API Error:', error);
        showToast('Terjadi kesalahan saat menghubungi server', 'error');
        throw error;
    }
}

// Load Agenda Data
async function loadAgendaData() {
    showLoading(true);
    try {
        const response = await apiRequest('agenda.php');
        currentData = response.data || [];
        renderTable(currentData);
        updateStats();
    } catch (error) {
        console.error('Error loading data:', error);
        showToast('Gagal memuat data', 'error');
    } finally {
        showLoading(false);
    }
}

// Render Table
function renderTable(data) {
    if (data.length === 0) {
        scheduleTableBody.innerHTML = '';
        noDataMessage.style.display = 'block';
        return;
    }
    
    noDataMessage.style.display = 'none';
    
    scheduleTableBody.innerHTML = data.map(item => `
        <tr>
            <td>${item.id}</td>
            <td>${formatDate(item.month, item.day)}</td>
            <td><strong>${escapeHtml(item.title)}</strong></td>
            <td>${escapeHtml(item.description)}</td>
            <td>
                <span class="status-badge ${item.is_active ? 'status-active' : 'status-inactive'}">
                    ${item.is_active ? 'Aktif' : 'Tidak Aktif'}
                </span>
            </td>
            <td>
                <div class="action-buttons">
                    <button class="btn btn-sm btn-warning" onclick="editAgenda(${item.id})">
                        <i class="fas fa-edit"></i> Edit
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteAgenda(${item.id})">
                        <i class="fas fa-trash"></i> Hapus
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Format Date
function formatDate(month, day) {
    const months = [
        'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
        'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
    ];
    return `${day} ${months[month - 1]}`;
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Update Statistics
function updateStats() {
    const totalEvents = currentData.length;
    const activeEvents = currentData.filter(item => item.is_active).length;
    const today = new Date();
    const upcomingEvents = currentData.filter(item => {
        const eventDate = new Date(today.getFullYear(), item.month - 1, item.day);
        return eventDate >= today;
    }).length;
    
    document.getElementById('totalEvents').textContent = totalEvents;
    document.getElementById('activeEvents').textContent = activeEvents;
    document.getElementById('upcomingEvents').textContent = upcomingEvents;
}

// Filter Data
function filterData() {
    const searchTerm = searchInput.value.toLowerCase();
    const selectedMonth = monthFilter.value;
    const selectedStatus = statusFilter.value;
    
    const filteredData = currentData.filter(item => {
        const matchesSearch = item.title.toLowerCase().includes(searchTerm) || 
                             item.description.toLowerCase().includes(searchTerm);
        const matchesMonth = !selectedMonth || item.month == selectedMonth;
        const matchesStatus = selectedStatus === '' || item.is_active == selectedStatus;
        
        return matchesSearch && matchesMonth && matchesStatus;
    });
    
    renderTable(filteredData);
}

// Modal Functions
function openAddModal() {
    document.getElementById('modalTitle').textContent = 'Tambah Jadwal Baru';
    scheduleForm.reset();
    document.getElementById('scheduleId').value = '';
    document.getElementById('statusText').textContent = 'Tidak Aktif';
    scheduleModal.classList.add('show');
}

function openEditModal(item) {
    document.getElementById('modalTitle').textContent = 'Edit Jadwal';
    document.getElementById('scheduleId').value = item.id;
    document.getElementById('month').value = item.month;
    document.getElementById('day').value = item.day;
    document.getElementById('title').value = item.title;
    document.getElementById('description').value = item.description;
    document.getElementById('isActive').checked = item.is_active;
    document.getElementById('statusText').textContent = item.is_active ? 'Aktif' : 'Tidak Aktif';
    scheduleModal.classList.add('show');
}

function closeModal() {
    scheduleModal.classList.remove('show');
    scheduleForm.reset();
}

function closeDeleteModal() {
    deleteModal.classList.remove('show');
    deleteId = null;
}

// CRUD Operations
async function handleFormSubmit(e) {
    e.preventDefault();
    
    const formData = {
        month: parseInt(document.getElementById('month').value),
        day: parseInt(document.getElementById('day').value),
        title: document.getElementById('title').value.trim(),
        description: document.getElementById('description').value.trim(),
        is_active: document.getElementById('isActive').checked ? 1 : 0
    };
    
    const id = document.getElementById('scheduleId').value;
    const isEdit = id !== '';
    
    try {
        showLoading(true);
        
        if (isEdit) {
            await apiRequest(`agenda.php?id=${id}`, 'PUT', formData);
            showToast('Agenda berhasil diperbarui', 'success');
        } else {
            await apiRequest('agenda.php', 'POST', formData);
            showToast('Agenda berhasil ditambahkan', 'success');
        }
        
        closeModal();
        await loadAgendaData();
    } catch (error) {
        console.error('Error saving data:', error);
        showToast(isEdit ? 'Gagal memperbarui agenda' : 'Gagal menambahkan agenda', 'error');
    } finally {
        showLoading(false);
    }
}

async function editAgenda(id) {
    const item = currentData.find(agenda => agenda.id === id);
    if (item) {
        openEditModal(item);
    }
}

function deleteAgenda(id) {
    deleteId = id;
    deleteModal.classList.add('show');
}

async function confirmDelete() {
    if (!deleteId) return;
    
    try {
        showLoading(true);
        await apiRequest(`agenda.php?id=${deleteId}`, 'DELETE');
        showToast('Agenda berhasil dihapus', 'success');
        closeDeleteModal();
        await loadAgendaData();
    } catch (error) {
        console.error('Error deleting data:', error);
        showToast('Gagal menghapus agenda', 'error');
    } finally {
        showLoading(false);
    }
}

// Loading State
function showLoading(show) {
    loadingSpinner.style.display = show ? 'block' : 'none';
}

// Toast Notification
function showToast(message, type = 'success') {
    toastMessage.textContent = message;
    toast.className = `toast ${type}`;
    
    // Set icon based on type
    const iconClasses = {
        success: 'fas fa-check-circle',
        error: 'fas fa-exclamation-circle',
        warning: 'fas fa-exclamation-triangle'
    };
    
    toastIcon.className = iconClasses[type] || iconClasses.success;
    
    // Show toast
    setTimeout(() => {
        toast.classList.add('show');
    }, 100);
    
    // Hide toast after 3 seconds
    setTimeout(() => {
        toast.classList.remove('show');
    }, 3000);
}

// Utility Functions
function validateForm() {
    const month = document.getElementById('month').value;
    const day = document.getElementById('day').value;
    const title = document.getElementById('title').value.trim();
    const description = document.getElementById('description').value.trim();
    
    if (!month) {
        showToast('Silakan pilih bulan', 'warning');
        return false;
    }
    
    if (!day || day < 1 || day > 31) {
        showToast('Silakan masukkan hari yang valid (1-31)', 'warning');
        return false;
    }
    
    if (!title) {
        showToast('Silakan masukkan judul', 'warning');
        return false;
    }
    
    if (!description) {
        showToast('Silakan masukkan deskripsi', 'warning');
        return false;
    }
    
    // Validate day based on month
    const daysInMonth = new Date(2024, month, 0).getDate();
    if (day > daysInMonth) {
        showToast(`Hari tidak valid untuk bulan yang dipilih (maksimal ${daysInMonth} hari)`, 'warning');
        return false;
    }
    
    return true;
}

// Export Functions (for potential future use)
function exportToCSV() {
    if (currentData.length === 0) {
        showToast('Tidak ada data untuk diekspor', 'warning');
        return;
    }
    
    const headers = ['ID', 'Bulan', 'Hari', 'Judul', 'Deskripsi', 'Status'];
    const rows = currentData.map(item => [
        item.id,
        item.month,
        item.day,
        `"${item.title}"`,
        `"${item.description}"`,
        item.is_active ? 'Aktif' : 'Tidak Aktif'
    ]);
    
    const csvContent = [headers, ...rows]
        .map(row => row.join(','))
        .join('\n');
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `jadwal_${new Date().toISOString().split('T')[0]}.csv`;
    a.click();
    window.URL.revokeObjectURL(url);
    
    showToast('Data berhasil diekspor ke CSV', 'success');
}

function printData() {
    window.print();
}

// Keyboard Shortcuts
document.addEventListener('keydown', function(event) {
    // Ctrl/Cmd + N: New schedule
    if ((event.ctrlKey || event.metaKey) && event.key === 'n') {
        event.preventDefault();
        openAddModal();
    }
    
    // Ctrl/Cmd + F: Focus search
    if ((event.ctrlKey || event.metaKey) && event.key === 'f') {
        event.preventDefault();
        searchInput.focus();
    }
    
    // Ctrl/Cmd + E: Export
    if ((event.ctrlKey || event.metaKey) && event.key === 'e') {
        event.preventDefault();
        exportToCSV();
    }
    
    // Ctrl/Cmd + P: Print
    if ((event.ctrlKey || event.metaKey) && event.key === 'p') {
        event.preventDefault();
        printData();
    }
});

// Auto-refresh (optional - every 5 minutes)
setInterval(async function() {
    try {
        const response = await apiRequest('agenda.php');
        const newData = response.data || [];
        
        // Only update if data has changed
        if (JSON.stringify(newData) !== JSON.stringify(currentData)) {
            currentData = newData;
            renderTable(currentData);
            updateStats();
            showToast('Agenda diperbarui', 'success');
        }
    } catch (error) {
        console.error('Auto-refresh error:', error);
    }
}, 300000); // 5 minutes