<?php
require_once 'config.php';

try {
    $conn = getConnection();
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'GET':
            handleGet($conn);
            break;
        case 'POST':
            handlePost($conn);
            break;
        case 'PUT':
            handlePut($conn);
            break;
        case 'DELETE':
            handleDelete($conn);
            break;
        default:
            sendResponse(false, 'Method not allowed', null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, 'Server error: ' . $e->getMessage(), null, 500);
}

// GET - Fetch all agenda or specific agenda
function handleGet($conn) {
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if ($id) {
        // Get specific agenda
        $stmt = $conn->prepare("SELECT * FROM agenda WHERE id = ?");
        $stmt->execute([$id]);
        $agenda = $stmt->fetch();
        
        if ($agenda) {
            sendResponse(true, 'Agenda found', $agenda);
        } else {
            sendResponse(false, 'Agenda not found', null, 404);
        }
    } else {
        // Get all agenda
        $stmt = $conn->prepare("SELECT * FROM agenda ORDER BY month, day, id");
        $stmt->execute();
        $agendaItems = $stmt->fetchAll();
        
        sendResponse(true, 'Agenda retrieved successfully', $agendaItems);
    }
}

// POST - Create new agenda
function handlePost($conn) {
    $data = getInputData();
    
    if (!$data) {
        sendResponse(false, 'Invalid JSON data', null, 400);
    }
    
    // Validate required fields
    $requiredFields = ['month', 'day', 'title', 'description'];
    if (!validateRequired($data, $requiredFields)) {
        sendResponse(false, 'Missing required fields', null, 400);
    }
    
    // Sanitize input
    $data = sanitizeInput($data);
    
    // Validate data
    $month = (int)$data['month'];
    $day = (int)$data['day'];
    $title = $data['title'];
    $description = $data['description'];
    $isActive = isset($data['is_active']) ? (int)$data['is_active'] : 0;
    
    // Validate month
    if ($month < 1 || $month > 12) {
        sendResponse(false, 'Invalid month. Must be between 1 and 12', null, 400);
    }
    
    // Validate day
    if ($day < 1 || $day > 31) {
        sendResponse(false, 'Invalid day. Must be between 1 and 31', null, 400);
    }
    
    // Validate day based on month
    $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, 2024);
    if ($day > $daysInMonth) {
        sendResponse(false, "Invalid day for month $month. Maximum is $daysInMonth days", null, 400);
    }
    
    // Validate title length
    if (strlen($title) > 255) {
        sendResponse(false, 'Title too long. Maximum 255 characters', null, 400);
    }
    
    // Insert into database
    try {
        $stmt = $conn->prepare("
            INSERT INTO agenda (month, day, title, description, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, NOW(), NOW())
        ");
        $stmt->execute([$month, $day, $title, $description, $isActive]);
        
        $newId = $conn->lastInsertId();
        
        // Get the inserted record
        $stmt = $conn->prepare("SELECT * FROM agenda WHERE id = ?");
        $stmt->execute([$newId]);
        $newAgenda = $stmt->fetch();
        
        sendResponse(true, 'Agenda created successfully', $newAgenda, 201);
        
    } catch (PDOException $e) {
        if ($e->getCode() == 23000) {
            sendResponse(false, 'Duplicate entry or constraint violation', null, 409);
        } else {
            sendResponse(false, 'Database error: ' . $e->getMessage(), null, 500);
        }
    }
}

// PUT - Update existing agenda
function handlePut($conn) {
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        sendResponse(false, 'Agenda ID is required', null, 400);
    }
    
    // Check if agenda exists
    $stmt = $conn->prepare("SELECT id FROM agenda WHERE id = ?");
    $stmt->execute([$id]);
    if (!$stmt->fetch()) {
        sendResponse(false, 'Agenda not found', null, 404);
    }
    
    $data = getInputData();
    
    if (!$data) {
        sendResponse(false, 'Invalid JSON data', null, 400);
    }
    
    // Validate required fields
    $requiredFields = ['month', 'day', 'title', 'description'];
    if (!validateRequired($data, $requiredFields)) {
        sendResponse(false, 'Missing required fields', null, 400);
    }
    
    // Sanitize input
    $data = sanitizeInput($data);
    
    // Validate data
    $month = (int)$data['month'];
    $day = (int)$data['day'];
    $title = $data['title'];
    $description = $data['description'];
    $isActive = isset($data['is_active']) ? (int)$data['is_active'] : 0;
    
    // Validate month
    if ($month < 1 || $month > 12) {
        sendResponse(false, 'Invalid month. Must be between 1 and 12', null, 400);
    }
    
    // Validate day
    if ($day < 1 || $day > 31) {
        sendResponse(false, 'Invalid day. Must be between 1 and 31', null, 400);
    }
    
    // Validate day based on month
    $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, 2024);
    if ($day > $daysInMonth) {
        sendResponse(false, "Invalid day for month $month. Maximum is $daysInMonth days", null, 400);
    }
    
    // Validate title length
    if (strlen($title) > 255) {
        sendResponse(false, 'Title too long. Maximum 255 characters', null, 400);
    }
    
    // Update database
    try {
        $stmt = $conn->prepare("
            UPDATE agenda 
            SET month = ?, day = ?, title = ?, description = ?, is_active = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $result = $stmt->execute([$month, $day, $title, $description, $isActive, $id]);
        
        if ($result) {
            // Get the updated record
            $stmt = $conn->prepare("SELECT * FROM agenda WHERE id = ?");
            $stmt->execute([$id]);
            $updatedAgenda = $stmt->fetch();
            
            sendResponse(true, 'Agenda updated successfully', $updatedAgenda);
        } else {
            sendResponse(false, 'Failed to update agenda', null, 500);
        }
        
    } catch (PDOException $e) {
        if ($e->getCode() == 23000) {
            sendResponse(false, 'Duplicate entry or constraint violation', null, 409);
        } else {
            sendResponse(false, 'Database error: ' . $e->getMessage(), null, 500);
        }
    }
}

// DELETE - Delete agenda
function handleDelete($conn) {
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        sendResponse(false, 'Agenda ID is required', null, 400);
    }
    
    // Check if agenda exists
    $stmt = $conn->prepare("SELECT * FROM agenda WHERE id = ?");
    $stmt->execute([$id]);
    $agenda = $stmt->fetch();
    
    if (!$agenda) {
        sendResponse(false, 'Agenda not found', null, 404);
    }
    
    try {
        $stmt = $conn->prepare("DELETE FROM agenda WHERE id = ?");
        $result = $stmt->execute([$id]);
        
        if ($result) {
            sendResponse(true, 'Agenda deleted successfully', $agenda);
        } else {
            sendResponse(false, 'Failed to delete agenda', null, 500);
        }
        
    } catch (PDOException $e) {
        sendResponse(false, 'Database error: ' . $e->getMessage(), null, 500);
    }
}
?>