// Load data from localStorage or use default
function loadData() {
    const savedData = localStorage.getItem('queenflorist_data');
    if (savedData) {
        try {
            const data = JSON.parse(savedData);
            return {
                products: data.products || defaultProducts,
                agendas: data.agendas || defaultAgendas
            };
        } catch (e) {
            console.error('Error loading data:', e);
        }
    }
    return { products: defaultProducts, agendas: defaultAgendas };
}

// Get image source (check localStorage first)
function getImageSrc(imagePath) {
    // Check if image is stored in localStorage
    const fileName = imagePath.split('/').pop();
    const storedImages = JSON.parse(localStorage.getItem('queenflorist_uploaded_images') || '{}');
    
    if (storedImages[fileName]) {
        return storedImages[fileName]; // Return base64 data
    }
    
    // Fallback to original path
    return imagePath;
}

// Default data
const defaultProducts = [
    {
        id: '1',
        name: 'Elegant Standing Rose',
        category: 'standing-flower',
        description: 'Standing flower dengan mawar merah segar yang elegan',
        image: 'https://i.imgur.com/fRauqUO.jpeg'
    },
    {
        id: '2',
        name: 'Paper Flower Art',
        category: 'paper-flower',
        description: 'Kertas bunga handmade yang indah dan tahan lama',
        image: 'assets/images/paper-flower.jpg'
    },
    {
        id: '3',
        name: 'Tur sympathy',
        category: 'duka-cita',
        description: 'Rangkaian bunga untuk ucapan duka cita',
        image: 'assets/images/sympathy.jpg'
    },
    {
        id: '4',
        name: 'Romantic Bouquet',
        category: 'bouquet',
        description: 'Buket bunga romantis untuk orang tersayang',
        image: 'assets/images/romantic-bouquet.jpg'
    },
    {
        id: '5',
        name: 'Wedding Car Premium',
        category: 'wedding-car',
        description: 'Dekorasi mobil pengantin premium',
        image: 'assets/images/wedding-car.jpg'
    },
    {
        id: '6',
        name: 'Birthday Balloon',
        category: 'balloon-decoration',
        description: 'Dekorasi balon ulang tahun meriah',
        image: 'assets/images/birthday-balloon.jpg'
    },
    {
        id: '7',
        name: 'Rustic Gift Box',
        category: 'kado-rustic',
        description: 'Kotak kado gaya rustic yang unik',
        image: 'assets/images/rustic-gift.jpg'
    },
    {
        id: '8',
        name: 'Balloon Parcel',
        category: 'balloon-parcel',
        description: 'Parcel balon yang menarik dan berwarna',
        image: 'assets/images/balloon-parcel.jpg'
    },
    {
        id: '9',
        name: 'Snack Flower Bouquet',
        category: 'snack-bouquet',
        description: 'Buket snack yang unik dan lezat',
        image: 'assets/images/snack-bouquet.jpg'
    },
    {
        id: '10',
        name: 'Tart Snack Box',
        category: 'tart-snack',
        description: 'Kue tart snack yang lezat',
        image: 'assets/images/tart-snack.jpg'
    }
];

const defaultAgendas = [
    {
        id: '1',
        date: '2024-10-14',
        description: 'Valentine Day Special - Diskon 20% untuk semua bouquet'
    },
    {
        id: '2',
        date: '2024-10-17',
        description: 'Workshop Flower Arrangement - Belajar merangkai bunga'
    },
    {
        id: '3',
        date: '2024-10-25',
        description: 'Mid Month Sale - Promo spesial standing flower'
    },
    {
        id: '4',
        date: '2024-12-14',
        description: 'Christmas Special - Diskon 20% untuk semua produk'
    },
    {
        id: '5',
        date: '2024-12-25',
        description: 'Christmas Day - Promo spesial standing flower'
    }
];

// Get current data
const currentData = loadData();
const products = currentData.products;
const agendas = currentData.agendas;

// Kategori produk
const categories = [
    { id: 'standing-flower', name: 'Standing Flower', icon: 'fa-spa', color: 'bg-pink-100 text-pink-600' },
    { id: 'paper-flower', name: 'Paper Flower', icon: 'fa-gift', color: 'bg-purple-100 text-purple-600' },
    { id: 'duka-cita', name: 'Duka Cita', icon: 'fa-heart', color: 'bg-gray-100 text-gray-600' },
    { id: 'bouquet', name: 'Bouquet', icon: 'fa-gift', color: 'bg-red-100 text-red-600' },
    { id: 'wedding-car', name: 'Wedding Car Decoration', icon: 'fa-car', color: 'bg-rose-100 text-rose-600' },
    { id: 'balloon-decoration', name: 'Balloon Decoration', icon: 'fa-circle', color: 'bg-blue-100 text-blue-600' },
    { id: 'kado-rustic', name: 'Kado Rustic', icon: 'fa-gift', color: 'bg-amber-100 text-amber-600' },
    { id: 'balloon-parcel', name: 'Balloon Parcel', icon: 'fa-circle', color: 'bg-cyan-100 text-cyan-600' },
    { id: 'snack-bouquet', name: 'Snack Bouquet', icon: 'fa-cookie', color: 'bg-orange-100 text-orange-600' },
    { id: 'tart-snack', name: 'Tart Snack', icon: 'fa-birthday-cake', color: 'bg-green-100 text-green-600' }
];

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    loadProducts();
    loadAgenda();
    updateCurrentMonth();
    
    // Check for admin access
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('admin') === 'true') {
        showAdminPanel();
    }
    
    // Auto refresh agenda every 30 seconds
    setInterval(() => {
        loadAgenda();
    }, 30000);
    
    // Listen for storage changes (when admin updates data)
    window.addEventListener('storage', function(e) {
        if (e.key === 'queenflorist_data') {
            console.log('Data updated, refreshing...');
            loadProducts();
            loadAgenda();
        }
    });
});

// Refresh data manually
function refreshData() {
    console.log('Manual refresh triggered');
    loadProducts();
    loadAgenda();
    showToast('Data berhasil diperbarui!');
}

// Show toast notification
function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `fixed bottom-4 right-4 px-4 py-2 rounded-lg shadow-lg z-50 ${
        type === 'success' ? 'bg-green-500' : 'bg-red-500'
    } text-white`;
    toast.textContent = message;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.remove();
    }, 3000);
}

// Load products
function loadProducts() {
    const grid = document.getElementById('productsGrid');
    grid.innerHTML = '';
    
    categories.forEach(category => {
        const categoryProducts = products.filter(p => p.category === category.id);
        
        const card = document.createElement('div');
        card.className = 'bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300';
        
        card.innerHTML = `
            <div class="p-4 text-center pb-3">
                <div class="${category.color} w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-3">
                    <i class="fas ${category.icon} text-2xl"></i>
                </div>
                <h3 class="text-lg font-semibold">${category.name}</h3>
            </div>
            <div class="px-4 pb-4">
                ${categoryProducts.length > 0 ? categoryProducts.map(product => `
                    <div class="cursor-pointer group" onclick="openOrderModal('${product.id}')">
                        <div class="aspect-square bg-gray-100 rounded-lg mb-3 overflow-hidden">
                            <img src="${getImageSrc(product.image)}" alt="${product.name}"
                                 class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                                 onerror="this.src='https://via.placeholder.com/300x300?text=No+Image'">
                        </div>
                        <h4 class="font-medium text-sm text-gray-800 group-hover:text-pink-600 transition-colors">
                            ${product.name}
                        </h4>
                        <p class="text-xs text-gray-500 mt-1 line-clamp-2">${product.description}</p>
                        <button class="w-full mt-3 bg-pink-500 hover:bg-pink-600 text-white py-2 rounded text-sm transition-colors">
                            Order
                        </button>
                    </div>
                `).join('') : `
                    <div class="text-center py-4">
                        <div class="aspect-square bg-gray-100 rounded-lg mb-3 flex items-center justify-center">
                            <i class="fas fa-spa text-3xl text-gray-300"></i>
                        </div>
                        <p class="text-sm text-gray-500">Belum ada produk</p>
                    </div>
                `}
            </div>
        `;
        
        grid.appendChild(card);
    });
}

// Load agenda
function loadAgenda() {
    // Reload data from localStorage to get latest updates
    const currentData = loadData();
    const agendas = currentData.agendas;
    
    const grid = document.getElementById('agendaGrid');
    const currentMonth = new Date().getMonth();
    const currentYear = new Date().getFullYear();
    
    console.log('Loading agendas:', agendas);
    console.log('Current month:', currentMonth, 'Current year:', currentYear);
    
    // Filter agenda untuk bulan sekarang
    const currentAgendas = agendas.filter(agenda => {
        const agendaDate = new Date(agenda.date);
        const matchesMonth = agendaDate.getMonth() === currentMonth;
        const matchesYear = agendaDate.getFullYear() === currentYear;
        console.log('Agenda:', agenda.description, 'Date:', agenda.date, 'Matches:', matchesMonth && matchesYear);
        return matchesMonth && matchesYear;
    });
    
    console.log('Current agendas for this month:', currentAgendas);
    
    grid.innerHTML = '';
    
    if (currentAgendas.length > 0) {
        // Sort by date
        currentAgendas.sort((a, b) => new Date(a.date) - new Date(b.date));
        
        currentAgendas.forEach(agenda => {
            const card = document.createElement('div');
            card.className = 'bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300';
            
            const date = new Date(agenda.date);
            const dateStr = date.toLocaleDateString('id-ID', { day: 'numeric', month: 'long' });
            const dayStr = date.toLocaleDateString('id-ID', { weekday: 'long' });
            
            card.innerHTML = `
                <div class="p-4">
                    <div class="flex items-center gap-3 mb-3">
                        <div class="bg-purple-100 p-3 rounded-full">
                            <i class="fas fa-calendar text-purple-600"></i>
                        </div>
                        <div>
                            <h3 class="text-lg font-semibold">${dateStr}</h3>
                            <p class="text-sm text-gray-500">${dayStr}</p>
                        </div>
                    </div>
                    <p class="text-gray-700">${agenda.description}</p>
                </div>
            `;
            
            grid.appendChild(card);
        });
    } else {
        grid.innerHTML = `
            <div class="col-span-full text-center py-8">
                <i class="fas fa-calendar text-4xl text-gray-300 mb-4"></i>
                <p class="text-gray-500">Belum ada agenda untuk bulan ini</p>
                <p class="text-sm text-gray-400 mt-2">Silakan tambah agenda melalui admin panel</p>
            </div>
        `;
    }
}

// Update current month display
function updateCurrentMonth() {
    const monthElement = document.getElementById('currentMonth');
    const currentMonth = new Date().toLocaleDateString('id-ID', { month: 'long' });
    monthElement.textContent = currentMonth;
}

// Open order modal
function openOrderModal(productId) {
    const product = products.find(p => p.id === productId);
    if (!product) return;
    
    const modal = document.getElementById('orderModal');
    const content = document.getElementById('modalContent');
    
    content.innerHTML = `
        <div class="space-y-4">
            <div class="aspect-square bg-gray-100 rounded-lg overflow-hidden">
                <img src="${getImageSrc(product.image)}" alt="${product.name}"
                     class="w-full h-full object-cover"
                     onerror="this.src='https://via.placeholder.com/300x300?text=No+Image'">
            </div>
            <div>
                <h4 class="font-semibold text-lg">${product.name}</h4>
                <p class="text-gray-600">${product.description}</p>
            </div>
            <div class="space-y-3">
                <div>
                    <label class="block text-sm font-medium mb-1">Nama Konsumen</label>
                    <input type="text" id="customerName" 
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-pink-500"
                           placeholder="Masukkan nama Anda">
                </div>
                <div>
                    <label class="block text-sm font-medium mb-1">Catatan</label>
                    <textarea id="customerNotes" rows="3"
                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-pink-500"
                              placeholder="Catatan khusus (opsional)"></textarea>
                </div>
                <button onclick="submitOrder('${product.id}')" 
                        class="w-full bg-green-500 hover:bg-green-600 text-white py-2 rounded-lg transition-colors">
                    <i class="fas fa-phone mr-2"></i>
                    Order via WhatsApp
                </button>
            </div>
        </div>
    `;
    
    modal.classList.remove('hidden');
    modal.classList.add('flex');
}

// Close order modal
function closeOrderModal() {
    const modal = document.getElementById('orderModal');
    modal.classList.add('hidden');
    modal.classList.remove('flex');
    
    // Clear form
    document.getElementById('customerName').value = '';
    document.getElementById('customerNotes').value = '';
}

// Submit order
function submitOrder(productId) {
    const product = products.find(p => p.id === productId);
    const name = document.getElementById('customerName').value;
    const notes = document.getElementById('customerNotes').value;
    
    if (!name.trim()) {
        alert('Silakan masukkan nama Anda');
        return;
    }
    
    const message = `Halo Queen Florist, saya ingin memesan:\n\n` +
        `Produk: ${product.name}\n` +
        `Kategori: ${categories.find(c => c.id === product.category)?.name}\n` +
        `Deskripsi: ${product.description}\n\n` +
        `Nama: ${name}\n` +
        `Catatan: ${notes || '-'}\n\n` +
        `Mohon informasi lebih lanjut mengenai produk ini. Terima kasih!`;
    
    const whatsappUrl = `https://wa.me/6281223968122?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
    
    closeOrderModal();
}

// Admin panel functions
function showAdminPanel() {
    window.location.href = 'admin.html';
}

function hideAdminPanel() {
    window.location.href = 'index.html';
}

function handleAdminLogin(event) {
    event.preventDefault();
    const password = document.getElementById('adminPassword').value;
    
    if (password === 'admin123') {
        window.location.href = 'admin.html';
    } else {
        alert('Invalid password');
    }
}

// Close modal when clicking outside
document.getElementById('orderModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeOrderModal();
    }
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeOrderModal();
    }
});