// Enhanced admin panel with better CRUD synchronization
class AdminDataManager {
    constructor() {
        this.storageKey = 'queenflorist_data';
        this.init();
    }

    init() {
        console.log('Admin data manager initializing...');
        this.loadData();
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Listen for storage changes from other tabs
        window.addEventListener('storage', (e) => {
            if (e.key === this.storageKey) {
                console.log('Admin: Storage change detected, reloading data...');
                this.loadData();
                this.refreshAllDisplays();
            }
        });

        // Listen for custom events
        window.addEventListener('dataUpdated', (e) => {
            console.log('Admin: Custom data update detected, reloading...');
            this.loadData();
            this.refreshAllDisplays();
        });
    }

    loadData() {
        try {
            const savedData = localStorage.getItem(this.storageKey);
            if (savedData) {
                const data = JSON.parse(savedData);
                window.products = data.products || window.defaultProducts;
                window.agendas = data.agendas || window.defaultAgendas;
                console.log('Admin: Data loaded successfully');
                console.log('Products:', window.products.length);
                console.log('Agendas:', window.agendas.length);
            } else {
                window.products = window.defaultProducts;
                window.agendas = window.defaultAgendas;
            }
        } catch (e) {
            console.error('Admin: Error loading data:', e);
            window.products = window.defaultProducts;
            window.agendas = window.defaultAgendas;
        }
    }

    saveData() {
        try {
            const data = {
                products: window.products,
                agendas: window.agendas,
                timestamp: Date.now(),
                lastUpdatedBy: 'admin'
            };
            
            localStorage.setItem(this.storageKey, JSON.stringify(data));
            
            // Trigger custom event for same-tab updates
            window.dispatchEvent(new CustomEvent('dataUpdated', {
                detail: data
            }));
            
            // Force storage event for cross-tab updates
            this.forceStorageEvent();
            
            console.log('Admin: Data saved and events triggered');
            console.log('Timestamp:', data.timestamp);
            
            return true;
        } catch (e) {
            console.error('Admin: Error saving data:', e);
            return false;
        }
    }

    forceStorageEvent() {
        // Create a temporary storage change to trigger storage event in other tabs
        const tempKey = this.storageKey + '_admin_temp';
        const tempValue = Date.now().toString();
        
        localStorage.setItem(tempKey, tempValue);
        localStorage.removeItem(tempKey);
        
        console.log('Admin: Storage event forced');
    }

    refreshAllDisplays() {
        console.log('Admin: Refreshing all displays...');
        
        // Refresh products if on products tab
        if (document.getElementById('productsList')) {
            this.loadProducts();
        }
        
        // Refresh agenda if on agenda tab
        if (document.getElementById('agendaList')) {
            this.loadAgenda();
        }
        
        // Show success notification
        this.showToast('Data berhasil diperbarui di semua tab!', 'success');
    }

    loadProducts() {
        const container = document.getElementById('productsList');
        if (!container) return;
        
        container.innerHTML = '';
        
        if (window.products.length === 0) {
            container.innerHTML = '<p class="text-gray-500 text-center py-4">Belum ada produk</p>';
            return;
        }
        
        window.products.forEach(product => {
            const item = document.createElement('div');
            item.className = 'border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow';
            
            item.innerHTML = `
                <div class="flex items-start gap-4">
                    <div class="aspect-square w-20 h-20 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                        <img src="${this.getImageSrc(product.image)}" alt="${product.name}"
                             class="w-full h-full object-cover"
                             onerror="this.src='https://via.placeholder.com/80x80?text=No+Image'">
                    </div>
                    <div class="flex-1">
                        <h3 class="font-semibold">${product.name}</h3>
                        <p class="text-sm text-gray-600">${product.description}</p>
                        <p class="text-xs text-gray-500 mt-1">Kategori: ${this.getCategoryName(product.category)}</p>
                    </div>
                    <div class="flex gap-2">
                        <button onclick="editProduct('${product.id}')" 
                                class="bg-blue-500 hover:bg-blue-600 text-white py-1 px-3 rounded text-sm transition-colors">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button onclick="deleteProduct('${product.id}')" 
                                class="bg-red-500 hover:bg-red-600 text-white py-1 px-3 rounded text-sm transition-colors">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
            `;
            
            container.appendChild(item);
        });
    }

    loadAgenda() {
        const container = document.getElementById('agendaList');
        if (!container) return;
        
        container.innerHTML = '';
        
        if (window.agendas.length === 0) {
            container.innerHTML = '<p class="text-gray-500 text-center py-4">Belum ada agenda</p>';
            return;
        }
        
        // Sort by date
        const sortedAgendas = [...window.agendas].sort((a, b) => new Date(a.date) - new Date(b.date));
        
        sortedAgendas.forEach(agenda => {
            const item = document.createElement('div');
            item.className = 'flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:shadow-md transition-shadow';
            
            const date = new Date(agenda.date);
            const dateStr = date.toLocaleDateString('id-ID', { 
                day: 'numeric', 
                month: 'long', 
                year: 'numeric' 
            });
            
            item.innerHTML = `
                <div class="flex-1">
                    <div class="font-semibold">${dateStr}</div>
                    <div class="text-sm text-gray-600">${agenda.description}</div>
                </div>
                <div class="flex gap-2">
                    <button onclick="editAgenda('${agenda.id}')" 
                            class="bg-blue-500 hover:bg-blue-600 text-white py-1 px-3 rounded text-sm transition-colors">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button onclick="deleteAgenda('${agenda.id}')" 
                            class="bg-red-500 hover:bg-red-600 text-white py-1 px-3 rounded text-sm transition-colors">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            `;
            
            container.appendChild(item);
        });
    }

    getImageSrc(imagePath) {
        const fileName = imagePath.split('/').pop();
        const storedImages = JSON.parse(localStorage.getItem('queenflorist_uploaded_images') || '{}');
        
        if (storedImages[fileName]) {
            return storedImages[fileName];
        }
        
        return imagePath;
    }

    getCategoryName(categoryId) {
        const categories = [
            { id: 'standing-flower', name: 'Standing Flower' },
            { id: 'paper-flower', name: 'Paper Flower' },
            { id: 'duka-cita', name: 'Duka Cita' },
            { id: 'bouquet', name: 'Bouquet' },
            { id: 'wedding-car', name: 'Wedding Car Decoration' },
            { id: 'balloon-decoration', name: 'Balloon Decoration' },
            { id: 'kado-rustic', name: 'Kado Rustic' },
            { id: 'balloon-parcel', name: 'Balloon Parcel' },
            { id: 'snack-bouquet', name: 'Snack Bouquet' },
            { id: 'tart-snack', name: 'Tart Snack' }
        ];
        
        const category = categories.find(c => c.id === categoryId);
        return category ? category.name : categoryId;
    }

    showToast(message, type = 'success') {
        const toast = document.getElementById('toast');
        const toastMessage = document.getElementById('toastMessage');
        
        if (!toast || !toastMessage) return;
        
        toastMessage.textContent = message;
        
        if (type === 'error') {
            toast.className = 'fixed bottom-4 right-4 bg-red-500 text-white px-4 py-2 rounded-lg shadow-lg z-50';
        } else {
            toast.className = 'fixed bottom-4 right-4 bg-green-500 text-white px-4 py-2 rounded-lg shadow-lg z-50';
        }
        
        toast.classList.remove('hidden');
        
        setTimeout(() => {
            toast.classList.add('hidden');
        }, 3000);
    }
}

// Initialize admin data manager
const adminDataManager = new AdminDataManager();

// Default data
window.defaultProducts = [
    {
        id: '1',
        name: 'Elegant Standing Rose',
        category: 'standing-flower',
        description: 'Standing flower dengan mawar merah segar yang elegan',
        image: 'assets/images/standing-rose.jpg'
    },
    {
        id: '2',
        name: 'Paper Flower Art',
        category: 'paper-flower',
        description: 'Kertas bunga handmade yang indah dan tahan lama',
        image: 'assets/images/paper-flower.jpg'
    },
    {
        id: '3',
        name: 'Tur sympathy',
        category: 'duka-cita',
        description: 'Rangkaian bunga untuk ucapan duka cita',
        image: 'assets/images/sympathy.jpg'
    },
    {
        id: '4',
        name: 'Romantic Bouquet',
        category: 'bouquet',
        description: 'Buket bunga romantis untuk orang tersayang',
        image: 'assets/images/romantic-bouquet.jpg'
    },
    {
        id: '5',
        name: 'Wedding Car Premium',
        category: 'wedding-car',
        description: 'Dekorasi mobil pengantin premium',
        image: 'assets/images/wedding-car.jpg'
    },
    {
        id: '6',
        name: 'Birthday Balloon',
        category: 'balloon-decoration',
        description: 'Dekorasi balon ulang tahun meriah',
        image: 'assets/images/birthday-balloon.jpg'
    },
    {
        id: '7',
        name: 'Rustic Gift Box',
        category: 'kado-rustic',
        description: 'Kotak kado gaya rustic yang unik',
        image: 'assets/images/rustic-gift.jpg'
    },
    {
        id: '8',
        name: 'Balloon Parcel',
        category: 'balloon-parcel',
        description: 'Parcel balon yang menarik dan berwarna',
        image: 'assets/images/balloon-parcel.jpg'
    },
    {
        id: '9',
        name: 'Snack Flower Bouquet',
        category: 'snack-bouquet',
        description: 'Buket snack yang unik dan lezat',
        image: 'assets/images/snack-bouquet.jpg'
    },
    {
        id: '10',
        name: 'Tart Snack Box',
        category: 'tart-snack',
        description: 'Kue tart snack yang lezat',
        image: 'assets/images/tart-snack.jpg'
    }
];

window.defaultAgendas = [
    {
        id: '1',
        date: '2024-10-14',
        description: 'Valentine Day Special - Diskon 20% untuk semua bouquet'
    },
    {
        id: '2',
        date: '2024-10-17',
        description: 'Workshop Flower Arrangement - Belajar merangkai bunga'
    },
    {
        id: '3',
        date: '2024-10-25',
        description: 'Mid Month Sale - Promo spesial standing flower'
    },
    {
        id: '4',
        date: '2024-12-14',
        description: 'Christmas Special - Diskon 20% untuk semua produk'
    },
    {
        id: '5',
        date: '2024-12-25',
        description: 'Christmas Day - Promo spesial standing flower'
    }
];

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    console.log('Admin panel initializing...');
    
    // Load data first
    adminDataManager.loadData();
    
    // Check if elements exist before loading
    const productsContainer = document.getElementById('productsList');
    const agendaContainer = document.getElementById('agendaList');
    
    if (productsContainer) {
        adminDataManager.loadProducts();
        console.log('Products loaded');
    } else {
        console.error('Products container not found on initialization');
    }
    
    if (agendaContainer) {
        adminDataManager.loadAgenda();
        console.log('Agenda loaded');
    } else {
        console.error('Agenda container not found on initialization');
    }
    
    // Add sync status indicator
    addSyncStatusIndicator();
    
    console.log('Admin panel initialized successfully');
});

// Add sync status indicator
function addSyncStatusIndicator() {
    const indicator = document.createElement('div');
    indicator.id = 'syncStatus';
    indicator.className = 'fixed top-4 right-4 bg-green-500 text-white px-3 py-1 rounded-full text-sm shadow-lg z-50';
    indicator.innerHTML = '<i class="fas fa-sync-alt mr-1"></i>Synced';
    document.body.appendChild(indicator);
    
    // Update sync status when data changes
    window.addEventListener('dataUpdated', () => {
        indicator.innerHTML = '<i class="fas fa-sync-alt mr-1 animate-spin"></i>Syncing...';
        setTimeout(() => {
            indicator.innerHTML = '<i class="fas fa-sync-alt mr-1"></i>Synced';
        }, 1000);
    });
}

// Login handler
function handleLogin(event) {
    event.preventDefault();
    const password = document.getElementById('adminPassword')?.value;
    
    if (password === 'admin123') {
        document.getElementById('loginScreen')?.classList.add('hidden');
        document.getElementById('adminDashboard')?.classList.remove('hidden');
        adminDataManager.showToast('Login successful!');
    } else {
        adminDataManager.showToast('Invalid password!', 'error');
    }
}

// Logout handler
function handleLogout() {
    if (confirm('Apakah Anda yakin ingin logout?')) {
        document.getElementById('loginScreen')?.classList.remove('hidden');
        document.getElementById('adminDashboard')?.classList.add('hidden');
        const passwordInput = document.getElementById('adminPassword');
        if (passwordInput) passwordInput.value = '';
        adminDataManager.showToast('Logged out successfully!');
    }
}

// Tab switching
function switchTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active state from all buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('text-pink-600', 'border-b-2', 'border-pink-600');
        btn.classList.add('text-gray-600');
    });
    
    // Show selected tab
    const selectedTab = document.getElementById(tabName + 'Tab');
    if (selectedTab) {
        selectedTab.classList.add('active');
    }
    
    // Add active state to clicked button
    event.target.classList.remove('text-gray-600');
    event.target.classList.add('text-pink-600', 'border-b-2', 'border-pink-600');
    
    // Load data for the active tab
    if (tabName === 'products') {
        adminDataManager.loadProducts();
    } else if (tabName === 'agenda') {
        adminDataManager.loadAgenda();
    }
}

// Enhanced product management
function handleProductSubmit(event) {
    event.preventDefault();
    
    const productId = document.getElementById('productId')?.value;
    const productData = {
        name: document.getElementById('productName')?.value || '',
        category: document.getElementById('productCategory')?.value || '',
        description: document.getElementById('productDescription')?.value || '',
        image: document.getElementById('productImage')?.value || 'assets/images/placeholder.jpg'
    };
    
    console.log('Product submit:', productId, productData);
    
    if (productId) {
        // Update existing product
        const index = window.products.findIndex(p => p.id === productId);
        if (index !== -1) {
            window.products[index] = { ...window.products[index], ...productData };
            adminDataManager.showToast('Produk berhasil diperbarui!');
        }
    } else {
        // Add new product
        const newProduct = {
            id: Date.now().toString(),
            ...productData
        };
        window.products.push(newProduct);
        adminDataManager.showToast('Produk berhasil ditambahkan!');
    }
    
    // Save data and trigger sync
    if (adminDataManager.saveData()) {
        adminDataManager.loadProducts();
        resetProductForm();
    }
}

function editProduct(productId) {
    const product = window.products.find(p => p.id === productId);
    if (!product) return;
    
    document.getElementById('productId').value = product.id;
    document.getElementById('productName').value = product.name;
    document.getElementById('productCategory').value = product.category;
    document.getElementById('productDescription').value = product.description;
    document.getElementById('productImage').value = product.image;
    
    document.getElementById('productFormTitle').textContent = 'Edit Produk';
    document.getElementById('productSubmitBtn').textContent = 'Update Produk';
    
    // Scroll to form
    document.getElementById('productsTab')?.scrollIntoView({ behavior: 'smooth' });
}

function deleteProduct(productId) {
    console.log('Deleting product:', productId);
    if (confirm('Apakah Anda yakin ingin menghapus produk ini?')) {
        try {
            const originalLength = window.products.length;
            window.products = window.products.filter(p => p.id !== productId);
            
            if (window.products.length < originalLength) {
                if (adminDataManager.saveData()) {
                    adminDataManager.loadProducts();
                    adminDataManager.showToast('Produk berhasil dihapus!');
                    console.log('Product deleted:', productId);
                }
            } else {
                adminDataManager.showToast('Produk tidak ditemukan!', 'error');
                console.error('Product not found for deletion:', productId);
            }
        } catch (error) {
            console.error('Error deleting product:', error);
            adminDataManager.showToast('Terjadi kesalahan saat menghapus produk!', 'error');
        }
    }
}

function resetProductForm() {
    try {
        document.getElementById('productId').value = '';
        document.getElementById('productName').value = '';
        document.getElementById('productCategory').value = '';
        document.getElementById('productDescription').value = '';
        document.getElementById('productImage').value = '';
        
        document.getElementById('productFormTitle').textContent = 'Tambah Produk Baru';
        document.getElementById('productSubmitBtn').textContent = 'Tambah Produk';
        console.log('Product form reset');
    } catch (error) {
        console.error('Error resetting product form:', error);
    }
}

// Enhanced agenda management
function handleAgendaSubmit(event) {
    event.preventDefault();
    
    const agendaId = document.getElementById('agendaId')?.value;
    const agendaData = {
        date: document.getElementById('agendaDate')?.value || '',
        description: document.getElementById('agendaDescription')?.value || ''
    };
    
    console.log('Agenda submit:', agendaId, agendaData);
    
    if (agendaId) {
        // Update existing agenda
        const index = window.agendas.findIndex(a => a.id === agendaId);
        if (index !== -1) {
            window.agendas[index] = { ...window.agendas[index], ...agendaData };
            adminDataManager.showToast('Agenda berhasil diperbarui!');
        }
    } else {
        // Add new agenda
        const newAgenda = {
            id: Date.now().toString(),
            ...agendaData
        };
        window.agendas.push(newAgenda);
        adminDataManager.showToast('Agenda berhasil ditambahkan!');
    }
    
    // Save data and trigger sync
    if (adminDataManager.saveData()) {
        adminDataManager.loadAgenda();
        resetAgendaForm();
    }
}

function editAgenda(agendaId) {
    const agenda = window.agendas.find(a => a.id === agendaId);
    if (!agenda) return;
    
    document.getElementById('agendaId').value = agenda.id;
    document.getElementById('agendaDate').value = agenda.date;
    document.getElementById('agendaDescription').value = agenda.description;
    
    document.getElementById('agendaFormTitle').textContent = 'Edit Agenda';
    document.getElementById('agendaSubmitBtn').textContent = 'Update Agenda';
    
    // Scroll to form
    document.getElementById('agendaTab')?.scrollIntoView({ behavior: 'smooth' });
}

function deleteAgenda(agendaId) {
    if (confirm('Apakah Anda yakin ingin menghapus agenda ini?')) {
        window.agendas = window.agendas.filter(a => a.id !== agendaId);
        
        if (adminDataManager.saveData()) {
            adminDataManager.loadAgenda();
            adminDataManager.showToast('Agenda berhasil dihapus!');
        }
    }
}

function resetAgendaForm() {
    document.getElementById('agendaId').value = '';
    document.getElementById('agendaDate').value = '';
    document.getElementById('agendaDescription').value = '';
    
    document.getElementById('agendaFormTitle').textContent = 'Tambah Agenda Baru';
    document.getElementById('agendaSubmitBtn').textContent = 'Tambah Agenda';
}

// Export data function
function exportData() {
    const data = {
        products: window.products,
        agendas: window.agendas,
        exportDate: new Date().toISOString()
    };
    
    const dataStr = JSON.stringify(data, null, 2);
    const dataBlob = new Blob([dataStr], { type: 'application/json' });
    const url = URL.createObjectURL(dataBlob);
    
    const link = document.createElement('a');
    link.href = url;
    link.download = `queenflorist-data-${new Date().toISOString().split('T')[0]}.json`;
    link.click();
    
    URL.revokeObjectURL(url);
    adminDataManager.showToast('Data berhasil diekspor!');
}

// Debug function
function debugAdminData() {
    console.log('=== Admin Debug Info ===');
    console.log('Products:', window.products);
    console.log('Agendas:', window.agendas);
    console.log('localStorage:', localStorage.getItem('queenflorist_data'));
    return {
        products: window.products,
        agendas: window.agendas
    };
}