// Data storage (simulasi database)
let products = [
    {
        id: '1',
        name: 'Elegant Standing Rose',
        category: 'standing-flower',
        description: 'Standing flower dengan mawar merah segar yang elegan',
        image: 'https://i.imgur.com/fRauqUO.jpeg'
    },
    {
        id: '2',
        name: 'Paper Flower Art',
        category: 'paper-flower',
        description: 'Kertas bunga handmade yang indah dan tahan lama',
        image: 'assets/images/paper-flower.jpg'
    },
    {
        id: '3',
        name: 'Tur sympathy',
        category: 'duka-cita',
        description: 'Rangkaian bunga untuk ucapan duka cita',
        image: 'assets/images/sympathy.jpg'
    },
    {
        id: '4',
        name: 'Romantic Bouquet',
        category: 'bouquet',
        description: 'Buket bunga romantis untuk orang tersayang',
        image: 'assets/images/romantic-bouquet.jpg'
    },
    {
        id: '5',
        name: 'Wedding Car Premium',
        category: 'wedding-car',
        description: 'Dekorasi mobil pengantin premium',
        image: 'assets/images/wedding-car.jpg'
    },
    {
        id: '6',
        name: 'Birthday Balloon',
        category: 'balloon-decoration',
        description: 'Dekorasi balon ulang tahun meriah',
        image: 'assets/images/birthday-balloon.jpg'
    },
    {
        id: '7',
        name: 'Rustic Gift Box',
        category: 'kado-rustic',
        description: 'Kotak kado gaya rustic yang unik',
        image: 'assets/images/rustic-gift.jpg'
    },
    {
        id: '8',
        name: 'Balloon Parcel',
        category: 'balloon-parcel',
        description: 'Parcel balon yang menarik dan berwarna',
        image: 'assets/images/balloon-parcel.jpg'
    },
    {
        id: '9',
        name: 'Snack Flower Bouquet',
        category: 'snack-bouquet',
        description: 'Buket snack yang unik dan lezat',
        image: 'assets/images/snack-bouquet.jpg'
    },
    {
        id: '10',
        name: 'Tart Snack Box',
        category: 'tart-snack',
        description: 'Kue tart snack yang lezat',
        image: 'assets/images/tart-snack.jpg'
    }
];

let agendas = [
    {
        id: '1',
        date: '2024-10-14',
        description: 'Valentine Day Special - Diskon 20% untuk semua bouquet'
    },
    {
        id: '2',
        date: '2024-10-17',
        description: 'Workshop Flower Arrangement - Belajar merangkai bunga'
    },
    {
        id: '3',
        date: '2024-10-25',
        description: 'Mid Month Sale - Promo spesial standing flower'
    },
    {
        id: '4',
        date: '2024-12-14',
        description: 'Christmas Special - Diskon 20% untuk semua produk'
    },
    {
        id: '5',
        date: '2024-12-25',
        description: 'Christmas Day - Promo spesial standing flower'
    }
];

// Load data from localStorage
function loadData() {
    const savedData = localStorage.getItem('queenflorist_data');
    if (savedData) {
        try {
            const data = JSON.parse(savedData);
            products = data.products || products;
            agendas = data.agendas || agendas;
        } catch (e) {
            console.error('Error loading data:', e);
        }
    }
}

// Save data to localStorage
function saveData() {
    const data = {
        products: products,
        agendas: agendas
    };
    localStorage.setItem('queenflorist_data', JSON.stringify(data));
    
    // Trigger storage event to notify other tabs
    window.dispatchEvent(new StorageEvent('storage', {
        key: 'queenflorist_data',
        newValue: JSON.stringify(data),
        oldValue: localStorage.getItem('queenflorist_data')
    }));
    
    console.log('Data saved and storage event triggered');
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    console.log('Admin panel initializing...');
    
    // Load data first
    loadData();
    
    // Check if elements exist before loading
    const productsContainer = document.getElementById('productsList');
    const agendaContainer = document.getElementById('agendaList');
    
    if (productsContainer) {
        loadProducts();
        console.log('Products loaded');
    } else {
        console.error('Products container not found on initialization');
    }
    
    if (agendaContainer) {
        loadAgenda();
        console.log('Agenda loaded');
    } else {
        console.error('Agenda container not found on initialization');
    }
    
    // Add debug info
    console.log('Total products:', products.length);
    console.log('Total agendas:', agendas.length);
    console.log('Admin panel initialized successfully');
});

// Debug function to test CRUD operations
function testCRUD() {
    console.log('=== CRUD Test ===');
    console.log('Current products:', products);
    console.log('Current agendas:', agendas);
    console.log('localStorage products:', localStorage.getItem('queenflorist_products'));
    console.log('localStorage agendas:', localStorage.getItem('queenflorist_agendas'));
}

// Login handler
function handleLogin(event) {
    event.preventDefault();
    const password = document.getElementById('adminPassword').value;
    
    if (password === 'admin123') {
        document.getElementById('loginScreen').classList.add('hidden');
        document.getElementById('adminDashboard').classList.remove('hidden');
        showToast('Login successful!');
    } else {
        showToast('Invalid password!', 'error');
    }
}

// Logout handler
function handleLogout() {
    if (confirm('Apakah Anda yakin ingin logout?')) {
        document.getElementById('loginScreen').classList.remove('hidden');
        document.getElementById('adminDashboard').classList.add('hidden');
        document.getElementById('adminPassword').value = '';
        showToast('Logged out successfully!');
    }
}

// Tab switching
function switchTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active state from all buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('text-pink-600', 'border-b-2', 'border-pink-600');
        btn.classList.add('text-gray-600');
    });
    
    // Show selected tab
    document.getElementById(tabName + 'Tab').classList.add('active');
    
    // Add active state to clicked button
    event.target.classList.remove('text-gray-600');
    event.target.classList.add('text-pink-600', 'border-b-2', 'border-pink-600');
}

// Image upload handling
function handleImageUpload(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    // Validate file type
    const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
    if (!validTypes.includes(file.type)) {
        showToast('Format file tidak valid! Gunakan JPG, PNG, atau WebP.', 'error');
        event.target.value = '';
        return;
    }
    
    // Validate file size (5MB max)
    const maxSize = 5 * 1024 * 1024; // 5MB
    if (file.size > maxSize) {
        showToast('Ukuran file terlalu besar! Max 5MB.', 'error');
        event.target.value = '';
        return;
    }
    
    // Create file reader for preview
    const reader = new FileReader();
    reader.onload = function(e) {
        // Show preview
        document.getElementById('previewImg').src = e.target.result;
        document.getElementById('imagePreview').classList.remove('hidden');
        document.getElementById('imageFileName').textContent = file.name;
        
        // Generate unique filename
        const timestamp = Date.now();
        const fileExtension = file.name.split('.').pop();
        const fileName = `${timestamp}.${fileExtension}`;
        
        // Save file path to hidden input
        document.getElementById('productImage').value = `assets/images/${fileName}`;
        
        // For XAMPP local environment, we'll store the base64 data
        // In a real server environment, this would upload to server
        storeImageFile(fileName, e.target.result);
        
        console.log('Image uploaded:', fileName);
    };
    
    reader.readAsDataURL(file);
}

function storeImageFile(fileName, base64Data) {
    // Store image data in localStorage for XAMPP environment
    // In production, this would be handled by server-side upload
    const storedImages = JSON.parse(localStorage.getItem('queenflorist_uploaded_images') || '{}');
    storedImages[fileName] = base64Data;
    localStorage.setItem('queenflorist_uploaded_images', JSON.stringify(storedImages));
}

function removeImage() {
    document.getElementById('productImageFile').value = '';
    document.getElementById('productImage').value = '';
    document.getElementById('imageFileName').textContent = 'Belum ada file dipilih';
    document.getElementById('imagePreview').classList.add('hidden');
    document.getElementById('previewImg').src = '';
}

function getImageSrc(imagePath) {
    // Check if image is stored in localStorage
    const fileName = imagePath.split('/').pop();
    const storedImages = JSON.parse(localStorage.getItem('queenflorist_uploaded_images') || '{}');
    
    if (storedImages[fileName]) {
        return storedImages[fileName]; // Return base64 data
    }
    
    // Fallback to original path
    return imagePath;
}

// Product management
function loadProducts() {
    const container = document.getElementById('productsList');
    if (!container) {
        console.error('Products container not found!');
        return;
    }
    container.innerHTML = '';
    
    if (products.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-4 col-span-full">Belum ada produk</p>';
        return;
    }
    
    products.forEach(product => {
        const card = document.createElement('div');
        card.className = 'border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow';
        
        const categoryName = getCategoryName(product.category);
        
        card.innerHTML = `
            <div class="aspect-square bg-gray-100 rounded-lg mb-3 overflow-hidden">
                <img src="${getImageSrc(product.image)}" alt="${product.name}"
                     class="w-full h-full object-cover"
                     onerror="this.src='https://via.placeholder.com/300x300?text=No+Image'; console.error('Image not found:', '${product.image}');">
            </div>
            <h3 class="font-semibold text-sm mb-1">${product.name}</h3>
            <p class="text-xs text-gray-500 mb-2">${categoryName}</p>
            <p class="text-xs text-gray-600 mb-3 line-clamp-2">${product.description}</p>
            <div class="flex gap-2">
                <button onclick="editProduct('${product.id}')" 
                        class="flex-1 bg-blue-500 hover:bg-blue-600 text-white py-1 px-2 rounded text-xs transition-colors">
                    <i class="fas fa-edit mr-1"></i>Edit
                </button>
                <button onclick="deleteProduct('${product.id}')" 
                        class="flex-1 bg-red-500 hover:bg-red-600 text-white py-1 px-2 rounded text-xs transition-colors">
                    <i class="fas fa-trash mr-1"></i>Hapus
                </button>
            </div>
        `;
        
        container.appendChild(card);
    });
    
    console.log('Loaded', products.length, 'products');
}

function getCategoryName(categoryId) {
    const categories = {
        'standing-flower': 'Standing Flower',
        'paper-flower': 'Paper Flower',
        'duka-cita': 'Duka Cita',
        'bouquet': 'Bouquet',
        'wedding-car': 'Wedding Car Decoration',
        'balloon-decoration': 'Balloon Decoration',
        'kado-rustic': 'Kado Rustic',
        'balloon-parcel': 'Balloon Parcel',
        'snack-bouquet': 'Snack Bouquet',
        'tart-snack': 'Tart Snack'
    };
    return categories[categoryId] || categoryId;
}

function handleProductSubmit(event) {
    event.preventDefault();
    console.log('Product form submitted');
    
    try {
        const productId = document.getElementById('productId').value;
        const productData = {
            name: document.getElementById('productName').value,
            category: document.getElementById('productCategory').value,
            description: document.getElementById('productDescription').value,
            image: document.getElementById('productImage').value
        };
        
        console.log('Product data:', productData);
        
        if (productId) {
            // Update existing product
            const index = products.findIndex(p => p.id === productId);
            if (index !== -1) {
                products[index] = { ...products[index], ...productData };
                showToast('Produk berhasil diperbarui!');
                console.log('Product updated:', productId);
            } else {
                showToast('Produk tidak ditemukan!', 'error');
                return;
            }
        } else {
            // Add new product
            const newProduct = {
                id: Date.now().toString(),
                ...productData
            };
            products.push(newProduct);
            showToast('Produk berhasil ditambahkan!');
            console.log('Product added:', newProduct.id);
        }
        
        saveData();
        loadProducts();
        resetProductForm();
        
    } catch (error) {
        console.error('Error submitting product:', error);
        showToast('Terjadi kesalahan saat menyimpan produk!', 'error');
    }
}

function editProduct(productId) {
    console.log('Editing product:', productId);
    const product = products.find(p => p.id === productId);
    if (!product) {
        console.error('Product not found:', productId);
        showToast('Produk tidak ditemukan!', 'error');
        return;
    }
    
    try {
        document.getElementById('productId').value = product.id;
        document.getElementById('productName').value = product.name;
        document.getElementById('productCategory').value = product.category;
        document.getElementById('productDescription').value = product.description;
        document.getElementById('productImage').value = product.image;
        
        // Show existing image preview
        const imageSrc = getImageSrc(product.image);
        if (imageSrc && imageSrc !== product.image) {
            // Image is stored in localStorage
            document.getElementById('previewImg').src = imageSrc;
            document.getElementById('imagePreview').classList.remove('hidden');
            document.getElementById('imageFileName').textContent = product.image.split('/').pop();
        } else {
            // Image is from assets folder
            document.getElementById('previewImg').src = product.image;
            document.getElementById('imagePreview').classList.remove('hidden');
            document.getElementById('imageFileName').textContent = product.image.split('/').pop();
        }
        
        document.getElementById('productFormTitle').textContent = 'Edit Produk';
        document.getElementById('productSubmitBtn').textContent = 'Update Produk';
        
        // Scroll to form
        document.getElementById('productsTab').scrollIntoView({ behavior: 'smooth' });
        console.log('Product loaded for editing:', productId);
    } catch (error) {
        console.error('Error loading product for edit:', error);
        showToast('Terjadi kesalahan saat memuat produk!', 'error');
    }
}

function deleteProduct(productId) {
    console.log('Deleting product:', productId);
    if (confirm('Apakah Anda yakin ingin menghapus produk ini?')) {
        try {
            const originalLength = products.length;
            products = products.filter(p => p.id !== productId);
            
            if (products.length < originalLength) {
                saveData();
                loadProducts();
                showToast('Produk berhasil dihapus!');
                console.log('Product deleted:', productId);
            } else {
                showToast('Produk tidak ditemukan!', 'error');
                console.error('Product not found for deletion:', productId);
            }
        } catch (error) {
            console.error('Error deleting product:', error);
            showToast('Terjadi kesalahan saat menghapus produk!', 'error');
        }
    }
}

function resetProductForm() {
    try {
        document.getElementById('productId').value = '';
        document.getElementById('productName').value = '';
        document.getElementById('productCategory').value = '';
        document.getElementById('productDescription').value = '';
        
        // Reset image upload
        removeImage();
        
        document.getElementById('productFormTitle').textContent = 'Tambah Produk Baru';
        document.getElementById('productSubmitBtn').textContent = 'Tambah Produk';
        console.log('Product form reset');
    } catch (error) {
        console.error('Error resetting product form:', error);
    }
}

// Agenda management
function loadAgenda() {
    const container = document.getElementById('agendaList');
    container.innerHTML = '';
    
    if (agendas.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-4">Belum ada agenda</p>';
        return;
    }
    
    // Sort by date
    const sortedAgendas = [...agendas].sort((a, b) => new Date(a.date) - new Date(b.date));
    
    sortedAgendas.forEach(agenda => {
        const item = document.createElement('div');
        item.className = 'flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:shadow-md transition-shadow';
        
        const date = new Date(agenda.date);
        const dateStr = date.toLocaleDateString('id-ID', { 
            day: 'numeric', 
            month: 'long', 
            year: 'numeric' 
        });
        
        item.innerHTML = `
            <div class="flex-1">
                <div class="font-semibold">${dateStr}</div>
                <div class="text-sm text-gray-600">${agenda.description}</div>
            </div>
            <div class="flex gap-2">
                <button onclick="editAgenda('${agenda.id}')" 
                        class="bg-blue-500 hover:bg-blue-600 text-white py-1 px-3 rounded text-sm transition-colors">
                    <i class="fas fa-edit"></i>
                </button>
                <button onclick="deleteAgenda('${agenda.id}')" 
                        class="bg-red-500 hover:bg-red-600 text-white py-1 px-3 rounded text-sm transition-colors">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `;
        
        container.appendChild(item);
    });
}

function handleAgendaSubmit(event) {
    event.preventDefault();
    
    const agendaId = document.getElementById('agendaId').value;
    const agendaData = {
        date: document.getElementById('agendaDate').value,
        description: document.getElementById('agendaDescription').value
    };
    
    if (agendaId) {
        // Update existing agenda
        const index = agendas.findIndex(a => a.id === agendaId);
        if (index !== -1) {
            agendas[index] = { ...agendas[index], ...agendaData };
            showToast('Agenda berhasil diperbarui!');
        }
    } else {
        // Add new agenda
        const newAgenda = {
            id: Date.now().toString(),
            ...agendaData
        };
        agendas.push(newAgenda);
        showToast('Agenda berhasil ditambahkan!');
    }
    
    saveData();
    loadAgenda();
    resetAgendaForm();
}

function editAgenda(agendaId) {
    const agenda = agendas.find(a => a.id === agendaId);
    if (!agenda) return;
    
    document.getElementById('agendaId').value = agenda.id;
    document.getElementById('agendaDate').value = agenda.date;
    document.getElementById('agendaDescription').value = agenda.description;
    
    document.getElementById('agendaFormTitle').textContent = 'Edit Agenda';
    document.getElementById('agendaSubmitBtn').textContent = 'Update Agenda';
    
    // Scroll to form
    document.getElementById('agendaTab').scrollIntoView({ behavior: 'smooth' });
}

function deleteAgenda(agendaId) {
    if (confirm('Apakah Anda yakin ingin menghapus agenda ini?')) {
        agendas = agendas.filter(a => a.id !== agendaId);
        saveData();
        loadAgenda();
        showToast('Agenda berhasil dihapus!');
    }
}

function resetAgendaForm() {
    document.getElementById('agendaId').value = '';
    document.getElementById('agendaDate').value = '';
    document.getElementById('agendaDescription').value = '';
    
    document.getElementById('agendaFormTitle').textContent = 'Tambah Agenda Baru';
    document.getElementById('agendaSubmitBtn').textContent = 'Tambah Agenda';
}

// Toast notification
function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    const toastMessage = document.getElementById('toastMessage');
    
    toastMessage.textContent = message;
    
    if (type === 'error') {
        toast.className = 'fixed bottom-4 right-4 bg-red-500 text-white px-4 py-2 rounded-lg shadow-lg';
    } else {
        toast.className = 'fixed bottom-4 right-4 bg-green-500 text-white px-4 py-2 rounded-lg shadow-lg';
    }
    
    toast.classList.remove('hidden');
    
    setTimeout(() => {
        toast.classList.add('hidden');
    }, 3000);
}

// Export data for main website
function exportData() {
    const data = {
        products: products,
        agendas: agendas
    };
    
    const dataStr = JSON.stringify(data, null, 2);
    const dataBlob = new Blob([dataStr], { type: 'application/json' });
    const url = URL.createObjectURL(dataBlob);
    
    const link = document.createElement('a');
    link.href = url;
    link.download = 'queenflorist-data.json';
    link.click();
    
    URL.revokeObjectURL(url);
    showToast('Data berhasil diekspor!');
}

// Add export button to header
document.addEventListener('DOMContentLoaded', function() {
    const header = document.querySelector('.flex.items-center.justify-between.py-4');
    const exportBtn = document.createElement('button');
    exportBtn.onclick = exportData;
    exportBtn.className = 'text-sm text-gray-600 hover:text-gray-800';
    exportBtn.innerHTML = '<i class="fas fa-download mr-1"></i> Export Data';
    header.appendChild(exportBtn);
});