-- Queen Florist Database Schema and Data
-- PHP MVC Version - Ready for CPanel Deployment
-- Generated: 2025-01-18

-- Create database if not exists
CREATE DATABASE IF NOT EXISTS `queen_florist` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Use the database
USE `queen_florist`;

-- Drop existing tables if they exist
DROP TABLE IF EXISTS `testimonials`;
DROP TABLE IF EXISTS `agenda`;
DROP TABLE IF EXISTS `products`;

-- Create Products table
CREATE TABLE `products` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `name` varchar(255) NOT NULL,
    `category` varchar(100) NOT NULL,
    `description` text DEFAULT NULL,
    `image_url` varchar(500) NOT NULL,
    `price` decimal(10,2) DEFAULT NULL,
    `is_best_seller` tinyint(1) NOT NULL DEFAULT 0,
    `icon` varchar(10) NOT NULL DEFAULT '🌸',
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_product_name` (`name`),
    KEY `idx_category` (`category`),
    KEY `idx_is_best_seller` (`is_best_seller`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Agenda table
CREATE TABLE `agenda` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `date` date NOT NULL,
    `title` varchar(255) NOT NULL,
    `description` text NOT NULL,
    `month` int(11) NOT NULL,
    `year` int(11) NOT NULL,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_date_title` (`date`, `title`),
    KEY `idx_date` (`date`),
    KEY `idx_month` (`month`),
    KEY `idx_year` (`year`),
    KEY `idx_month_year` (`month`, `year`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Create Testimonials table
CREATE TABLE `testimonials` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `name` varchar(255) NOT NULL,
    `message` text NOT NULL,
    `rating` int(11) NOT NULL,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_rating` (`rating`),
    KEY `idx_created_at` (`created_at`),
    KEY `idx_name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert Products data
INSERT INTO `products` (`id`, `name`, `category`, `description`, `image_url`, `price`, `is_best_seller`, `icon`, `created_at`, `updated_at`) VALUES
(1, 'Standing Flower', 'standing flower', 'Standing flower elegan untuk berbagai acara', 'https://i.imgur.com/fRauqUO.jpeg', 500000.00, 1, '🌹', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(2, 'Paper Flower', 'paper flower', 'Bunga kertas cantik dan awet', 'https://i.imgur.com/JqYHlaX.jpeg', 150000.00, 1, '🌸', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(3, 'Duka Cita', 'duka cita', 'Karangan bunga untuk ucapan duka cita', 'https://i.imgur.com/JuJhAyP.jpeg', 750000.00, 0, '🕊️', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(4, 'Bouquet', 'bouquet', 'Buket bunga segar untuk orang tersayang', 'https://i.imgur.com/hl90R43.jpeg', 350000.00, 1, '💐', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(5, 'Wedding Car Decoration', 'wedding car decoration', 'Dekorasi mobil pengantin yang romantis', 'https://i.imgur.com/zuueu4D.jpeg', 1500000.00, 0, '🚗', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(6, 'Balloon Decoration', 'balloon decoration', 'Dekorasi balon untuk berbagai pesta', 'https://i.imgur.com/mED89sx.jpeg', 800000.00, 0, '🎈', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(7, 'Kado Rustic', 'kado rustic', 'Kado dengan gaya rustic yang unik', 'https://i.imgur.com/Rja49Tr.jpeg', 250000.00, 0, '🎁', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(8, 'Balloon Parcel', 'ballon parcel', 'Parcel balon yang menarik', 'https://i.imgur.com/DPrCrrF.jpeg', 400000.00, 0, '📦', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(9, 'Snack Bouquet', 'snack bouquet', 'Buket snack yang lezat dan menarik', 'https://i.imgur.com/QmJMTVR.jpeg', 200000.00, 1, '🍿', '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(10, 'Tart Snack', 'tart snack', 'Tart snack untuk berbagai acara', 'https://i.imgur.com/9gzsAw7.jpeg', 300000.00, 0, '🧁', '2025-01-18 00:00:00', '2025-01-18 00:00:00');

-- Insert Agenda data (current year)
INSERT INTO `agenda` (`id`, `date`, `title`, `description`, `month`, `year`, `created_at`, `updated_at`) VALUES
(1, '2025-10-24', 'Hari Dokter Indonesia', 'Hari untuk menghormati jasa para dokter', 10, 2025, '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(2, '2025-10-27', 'Hari Penerbangan Nasional', 'Memperingati sejarah penerbangan Indonesia', 10, 2025, '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(3, '2025-10-28', 'Hari Sumpah Pemuda', 'Memperingati Sumpah Pemuda 1928', 10, 2025, '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(4, '2025-10-30', 'Hari Keuangan', 'Hari untuk meningkatkan literasi keuangan', 10, 2025, '2025-01-18 00:00:00', '2025-01-18 00:00:00');

-- Insert Testimonials data
INSERT INTO `testimonials` (`id`, `name`, `message`, `rating`, `created_at`, `updated_at`) VALUES
(1, 'Siti Nurhaliza', 'Bunga yang saya pesan sangat indah dan segar. Pelayanan Queen Florist sangat memuaskan!', 5, '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(2, 'Budi Santoso', 'Saya memesan bouquet untuk anniversary istri, hasilnya sangat bagus dan tepat waktu. Terima kasih Queen Florist!', 5, '2025-01-18 00:00:00', '2025-01-18 00:00:00'),
(3, 'Maya Putri', 'Standing flower untuk acara perusahaan kami sangat elegan. Recommended banget!', 5, '2025-01-18 00:00:00', '2025-01-18 00:00:00');

-- Create views for easier data access

-- View for products with category info
CREATE VIEW `v_products_details` AS
SELECT 
    p.*,
    CASE 
        WHEN p.is_best_seller = 1 THEN 'Best Seller'
        ELSE 'Regular'
    END as product_type
FROM products p;

-- View for current month agenda
CREATE VIEW `v_current_month_agenda` AS
SELECT 
    a.*,
    DAYNAME(a.date) as day_name,
    MONTHNAME(a.date) as month_name
FROM agenda a 
WHERE a.month = MONTH(CURRENT_DATE()) 
  AND a.year = YEAR(CURRENT_DATE())
ORDER BY a.date;

-- View for testimonials with rating text
CREATE VIEW `v_testimonials_rating` AS
SELECT 
    t.*,
    CASE 
        WHEN t.rating = 5 THEN 'Excellent'
        WHEN t.rating = 4 THEN 'Very Good'
        WHEN t.rating = 3 THEN 'Good'
        WHEN t.rating = 2 THEN 'Fair'
        ELSE 'Poor'
    END as rating_text
FROM testimonials t;

-- Create stored procedures for common operations

DELIMITER //

-- Procedure to get product statistics
CREATE PROCEDURE `GetProductStats`()
BEGIN
    SELECT 
        COUNT(*) as total_products,
        SUM(CASE WHEN is_best_seller = 1 THEN 1 ELSE 0 END) as best_sellers,
        COUNT(DISTINCT category) as categories,
        AVG(price) as average_price
    FROM products;
END//

-- Procedure to get agenda statistics
CREATE PROCEDURE `GetAgendaStats`()
BEGIN
    SELECT 
        COUNT(*) as total_agenda,
        SUM(CASE WHEN month = MONTH(CURRENT_DATE()) AND year = YEAR(CURRENT_DATE()) THEN 1 ELSE 0 END) as current_month_agenda,
        SUM(CASE WHEN date >= CURRENT_DATE() THEN 1 ELSE 0 END) as upcoming_agenda
    FROM agenda;
END//

-- Procedure to get testimonial statistics
CREATE PROCEDURE `GetTestimonialStats`()
BEGIN
    SELECT 
        COUNT(*) as total_testimonials,
        AVG(rating) as average_rating,
        SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star_reviews,
        SUM(CASE WHEN rating >= 4 THEN 1 ELSE 0 END) as positive_reviews
    FROM testimonials;
END//

DELIMITER ;

-- Create triggers for data integrity

-- Trigger to ensure month and year match date
DELIMITER //
CREATE TRIGGER `before_agenda_insert` 
BEFORE INSERT ON `agenda`
FOR EACH ROW
BEGIN
    SET NEW.month = MONTH(NEW.date);
    SET NEW.year = YEAR(NEW.date);
END//

CREATE TRIGGER `before_agenda_update` 
BEFORE UPDATE ON `agenda`
FOR EACH ROW
BEGIN
    SET NEW.month = MONTH(NEW.date);
    SET NEW.year = YEAR(NEW.date);
END//
DELIMITER ;

-- Create indexes for better performance
CREATE INDEX `idx_products_name` ON `products` (`name`);
CREATE INDEX `idx_agenda_title` ON `agenda` (`title`);
CREATE INDEX `idx_testimonials_name_message` ON `testimonials` (`name`, `message`(100));

-- Add foreign key constraints if needed (for future expansion)
-- ALTER TABLE `orders` ADD CONSTRAINT `fk_order_product` FOREIGN KEY (`product_id`) REFERENCES `products` (`id`) ON DELETE CASCADE;

-- Create user for application (optional)
-- CREATE USER 'queen_florist_user'@'localhost' IDENTIFIED BY 'secure_password';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON `queen_florist`.* TO 'queen_florist_user'@'localhost';
-- FLUSH PRIVILEGES;

-- Final setup confirmation
SELECT 'Queen Florist database setup completed successfully!' as status,
       (SELECT COUNT(*) FROM products) as products_count,
       (SELECT COUNT(*) FROM agenda) as agenda_count,
       (SELECT COUNT(*) FROM testimonials) as testimonials_count;