<?php
require_once 'config/database.php';

/**
 * Testimonial Model
 * Handles all testimonial-related database operations
 */
class TestimonialModel {
    private $conn;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
    }
    
    /**
     * Get all testimonials
     * @return array
     */
    public function getAllTestimonials() {
        $sql = "SELECT * FROM testimonials ORDER BY created_at DESC";
        $result = $this->conn->query($sql);
        
        $testimonials = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $testimonials[] = $row;
            }
        }
        
        return $testimonials;
    }
    
    /**
     * Get testimonial by ID
     * @param int $id
     * @return array|null
     */
    public function getTestimonialById($id) {
        $sql = "SELECT * FROM testimonials WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return $result->fetch_assoc();
        }
        
        return null;
    }
    
    /**
     * Get featured testimonials (highest rated)
     * @param int $limit
     * @return array
     */
    public function getFeaturedTestimonials($limit = 3) {
        $sql = "SELECT * FROM testimonials WHERE rating >= 4 ORDER BY rating DESC, created_at DESC LIMIT ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $testimonials = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $testimonials[] = $row;
            }
        }
        
        return $testimonials;
    }
    
    /**
     * Get testimonials by rating
     * @param int $rating
     * @return array
     */
    public function getTestimonialsByRating($rating) {
        $sql = "SELECT * FROM testimonials WHERE rating = ? ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $rating);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $testimonials = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $testimonials[] = $row;
            }
        }
        
        return $testimonials;
    }
    
    /**
     * Create new testimonial
     * @param array $data
     * @return bool
     */
    public function createTestimonial($data) {
        $sql = "INSERT INTO testimonials (name, message, rating) VALUES (?, ?, ?)";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ssi", 
            $data['name'], 
            $data['message'], 
            $data['rating']
        );
        
        return $stmt->execute();
    }
    
    /**
     * Update testimonial
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateTestimonial($id, $data) {
        $sql = "UPDATE testimonials SET name = ?, message = ?, rating = ?, 
                updated_at = CURRENT_TIMESTAMP WHERE id = ?";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ssii", 
            $data['name'], 
            $data['message'], 
            $data['rating'],
            $id
        );
        
        return $stmt->execute();
    }
    
    /**
     * Delete testimonial
     * @param int $id
     * @return bool
     */
    public function deleteTestimonial($id) {
        $sql = "DELETE FROM testimonials WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        
        return $stmt->execute();
    }
    
    /**
     * Get average rating
     * @return float
     */
    public function getAverageRating() {
        $sql = "SELECT AVG(rating) as average_rating FROM testimonials";
        $result = $this->conn->query($sql);
        $row = $result->fetch_assoc();
        
        return $row['average_rating'] ? round($row['average_rating'], 1) : 0;
    }
    
    /**
     * Get rating distribution
     * @return array
     */
    public function getRatingDistribution() {
        $sql = "SELECT rating, COUNT(*) as count FROM testimonials GROUP BY rating ORDER BY rating DESC";
        $result = $this->conn->query($sql);
        
        $distribution = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $distribution[$row['rating']] = $row['count'];
            }
        }
        
        return $distribution;
    }
    
    /**
     * Count total testimonials
     * @return int
     */
    public function countTestimonials() {
        $sql = "SELECT COUNT(*) as total FROM testimonials";
        $result = $this->conn->query($sql);
        $row = $result->fetch_assoc();
        
        return $row['total'];
    }
    
    /**
     * Search testimonials
     * @param string $keyword
     * @return array
     */
    public function searchTestimonials($keyword) {
        $sql = "SELECT * FROM testimonials WHERE 
                name LIKE ? OR message LIKE ? ORDER BY created_at DESC";
        
        $searchTerm = "%" . $keyword . "%";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ss", $searchTerm, $searchTerm);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $testimonials = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $testimonials[] = $row;
            }
        }
        
        return $testimonials;
    }
}
?>