<?php
require_once __DIR__ . '/../../config/database.php';

class AdminModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Login authentication
     */
    public function login($username, $password) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, username, password, email, full_name, role, is_active 
                FROM admin_users 
                WHERE username = :username AND is_active = 1
            ");
            $stmt->execute(['username' => $username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && password_verify($password, $user['password'])) {
                // Update last login
                $this->updateLastLogin($user['id']);
                unset($user['password']); // Remove password from response
                return $user;
            }
            return false;
        } catch (PDOException $e) {
            error_log("Login error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update last login time
     */
    private function updateLastLogin($adminId) {
        try {
            $stmt = $this->db->prepare("
                UPDATE admin_users 
                SET last_login = NOW() 
                WHERE id = :id
            ");
            $stmt->execute(['id' => $adminId]);
        } catch (PDOException $e) {
            error_log("Update last login error: " . $e->getMessage());
        }
    }
    
    /**
     * Get admin by ID
     */
    public function getAdminById($id) {
        try {
            $stmt = $this->db->prepare("
                SELECT id, username, email, full_name, role, is_active, last_login, created_at 
                FROM admin_users 
                WHERE id = :id
            ");
            $stmt->execute(['id' => $id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get admin error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Log admin activity
     */
    public function logActivity($adminId, $action, $tableName = null, $recordId = null, $description = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO admin_activity_logs 
                (admin_id, action, table_name, record_id, description, ip_address, user_agent) 
                VALUES (:admin_id, :action, :table_name, :record_id, :description, :ip_address, :user_agent)
            ");
            
            $stmt->execute([
                'admin_id' => $adminId,
                'action' => $action,
                'table_name' => $tableName,
                'record_id' => $recordId,
                'description' => $description,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            error_log("Log activity error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get activity logs
     */
    public function getActivityLogs($limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT al.*, au.username 
                FROM admin_activity_logs al
                LEFT JOIN admin_users au ON al.admin_id = au.id
                ORDER BY al.created_at DESC 
                LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get activity logs error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get admin statistics
     */
    public function getStats() {
        try {
            $stmt = $this->db->prepare("SELECT * FROM admin_stats");
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get stats error: " . $e->getMessage());
            return [
                'total_products' => 0,
                'total_agenda' => 0,
                'active_admins' => 0,
                'today_activities' => 0
            ];
        }
    }
    
    /**
     * Change password
     */
    public function changePassword($adminId, $currentPassword, $newPassword) {
        try {
            // Verify current password
            $stmt = $this->db->prepare("SELECT password FROM admin_users WHERE id = :id");
            $stmt->execute(['id' => $adminId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user || !password_verify($currentPassword, $user['password'])) {
                return false;
            }
            
            // Update password
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $this->db->prepare("
                UPDATE admin_users 
                SET password = :password, updated_at = NOW() 
                WHERE id = :id
            ");
            $stmt->execute([
                'password' => $hashedPassword,
                'id' => $adminId
            ]);
            
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Change password error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create new admin user
     */
    public function createAdmin($data) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO admin_users (username, password, email, full_name, role) 
                VALUES (:username, :password, :email, :full_name, :role)
            ");
            
            $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
            
            $stmt->execute([
                'username' => $data['username'],
                'password' => $hashedPassword,
                'email' => $data['email'],
                'full_name' => $data['full_name'],
                'role' => $data['role'] ?? 'editor'
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            error_log("Create admin error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all admin users
     */
    public function getAllAdmins() {
        try {
            $stmt = $this->db->prepare("
                SELECT id, username, email, full_name, role, is_active, last_login, created_at 
                FROM admin_users 
                ORDER BY created_at DESC
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get all admins error: " . $e->getMessage());
            return [];
        }
    }
}