<?php
require_once __DIR__ . '/../../config/database.php';

/**
 * Admin Agenda Model
 * Handles all agenda-related database operations for admin panel
 */
class AdminAgendaModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all agenda items with pagination
     */
    public function getAllAgenda($limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT a.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM agenda a
                LEFT JOIN admin_users au ON a.created_by = au.id
                LEFT JOIN admin_users au2 ON a.updated_by = au2.id
                ORDER BY a.date DESC 
                LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get all agenda error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get agenda by ID
     */
    public function getAgendaById($id) {
        try {
            $stmt = $this->db->prepare("
                SELECT a.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM agenda a
                LEFT JOIN admin_users au ON a.created_by = au.id
                LEFT JOIN admin_users au2 ON a.updated_by = au2.id
                WHERE a.id = :id
            ");
            $stmt->execute(['id' => $id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get agenda by ID error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create new agenda item
     */
    public function createAgenda($data, $adminId = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO agenda (date, title, description, month, year, created_by) 
                VALUES (:date, :title, :description, :month, :year, :created_by)
            ");
            
            $stmt->execute([
                'date' => $data['date'],
                'title' => $data['title'],
                'description' => $data['description'],
                'month' => date('n', strtotime($data['date'])),
                'year' => date('Y', strtotime($data['date'])),
                'created_by' => $adminId
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            error_log("Create agenda error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update agenda item
     */
    public function updateAgenda($id, $data, $adminId = null) {
        try {
            $stmt = $this->db->prepare("
                UPDATE agenda SET 
                    date = :date, 
                    title = :title, 
                    description = :description, 
                    month = :month, 
                    year = :year, 
                    updated_by = :updated_by,
                    updated_at = CURRENT_TIMESTAMP 
                WHERE id = :id
            ");
            
            $stmt->execute([
                'date' => $data['date'],
                'title' => $data['title'],
                'description' => $data['description'],
                'month' => date('n', strtotime($data['date'])),
                'year' => date('Y', strtotime($data['date'])),
                'updated_by' => $adminId,
                'id' => $id
            ]);
            
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Update agenda error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete agenda item
     */
    public function deleteAgenda($id) {
        try {
            $stmt = $this->db->prepare("DELETE FROM agenda WHERE id = :id");
            $stmt->execute(['id' => $id]);
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Delete agenda error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Count total agenda items
     */
    public function countAgenda() {
        try {
            $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM agenda");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result['total'];
        } catch (PDOException $e) {
            error_log("Count agenda error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Search agenda items
     */
    public function searchAgenda($keyword, $limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT a.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM agenda a
                LEFT JOIN admin_users au ON a.created_by = au.id
                LEFT JOIN admin_users au2 ON a.updated_by = au2.id
                WHERE a.title LIKE :keyword OR a.description LIKE :keyword OR a.date LIKE :keyword
                ORDER BY a.date DESC 
                LIMIT :limit OFFSET :offset
            ");
            
            $searchTerm = '%' . $keyword . '%';
            $stmt->bindValue(':keyword', $searchTerm);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Search agenda error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get upcoming agenda (from today onwards)
     */
    public function getUpcomingAgenda($limit = 10) {
        try {
            $today = date('Y-m-d');
            $stmt = $this->db->prepare("
                SELECT * FROM agenda 
                WHERE date >= :today 
                ORDER BY date ASC 
                LIMIT :limit
            ");
            $stmt->execute(['today' => $today, 'limit' => $limit]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get upcoming agenda error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get agenda by month and year
     */
    public function getAgendaByMonth($month, $year) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM agenda 
                WHERE month = :month AND year = :year 
                ORDER BY date ASC
            ");
            $stmt->execute(['month' => $month, 'year' => $year]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get agenda by month error: " . $e->getMessage());
            return [];
        }
    }
}